

/**************************************************************************************************************/
/* This SAS program illustrates the use of the mixtran and indivint macros.                                   */
/* The following files are also required to run this example:                                                 */
/*   mixtran_macro_v1.1.sas                                                                                   */
/*   indivint_macro_v2.1.sas                                                                                  */
/*   data.example.males12plus.xpt                                                                             */
/* The path must be modified in the following lines:                                                          */

*** Include the MIXTRAN and INDIVINT macros ***;
%include "&home\NCI_Method\Macros_v2.1\mixtran_macro_v2.1.sas" ;
%include "&home\NCI_Method\Macros_v2.1\indivint_macro_v2.1.sas" ;

*** Input data file ***;
filename tranfile "&home\NCI_Method\Data\data.example.males12plus.xpt";

*** Library for data sets output by the MIXTRAN macro ***;
libname mylib  "&home\NCI_Method\Examples_v2.1\example4";
/**************************************************************************************************************/



title1 "Example:  Total Mercury and Individual Usual Intake of Fish, NHANES 2003-2004, Males, Ages 12+";
title2 "NLMIXED Cov:  Race, Educ, Age, I(FFQ=0), BoxCox(FFQ)     Linear Reg Cov:  E( BoxCox(Ti,lamT) | Xi )";



************************************;
*** Covariates for nlmixed model ***;
************************************;

%let covnlmixmod = race1 race3 dmdeduc1 dmdeduc3 ridageyr ffqeqzero bcffq0mean;



***********************;
*** Weight variable ***;
***********************;

%let weight_var = rwthg;       



*********************************************************************************************************************;
*** Box-Cox parameter for transformation of individual usual intake.  Comparison usual intake values:  T0 and T1. ***;
*********************************************************************************************************************;

%let boxcox_t_param = 0.11;

%let t0 = 0.1;
%let t1 = 1; 



*********************************************;
*** Import the data file for this example ***;
*********************************************;

proc cimport data=examplemales12plus infile=tranfile;
run;



*********************************************;

data datamrec(keep = subjectid replicvar wave seqn numdays ridageyr race1 race3 dmdeduc1 dmdeduc3 
                     ffqeqzero bcffq0mean R repeat)
     subj1rec(keep = subjectid replicvar wave seqn numdays ridageyr race1 race3 dmdeduc1 dmdeduc3 
                     ffqeqzero bcffq0mean R1 R2 logtotmerc)
     ;
  set examplemales12plus;
  by subjectid;

  ****************************************;
  *** Replicate (i.e. weight) variable ***;
  ****************************************;

  replicvar = &weight_var;

  ********************************************;
  *** Output record for 1st 24-hour recall ***;
  ********************************************;

  if R1 >= 0 then do;

    repeat = 1;

    R = R1;

    output datamrec;

  end;
  
  ********************************************;
  *** Output record for 2nd 24-hour recall ***;
  ********************************************;
  
  if R2 >= 0 then do;

    repeat = 2;

    R = R2;

    output datamrec;

  end; 
  
  ****************************************;
  *** Output 1 record for each subject ***;
  ****************************************;
 
  output subj1rec;
  
run;
  


************************************************************;
*** Calculate the smallest positive 24-hour recall value ***;
************************************************************;

proc univariate data=datamrec noprint;
  where R>0;
  var R;
  output out=outmin_amt min=min_amt;
run;

************************************************************;


proc print data=datamrec(obs=20);
  var subjectid replicvar repeat R &covnlmixmod;
  title3 "Some Observations from the Data Set Used to Fit the Starting Value Models and the Two-Part Nonlinear Mixed Model";
run;
title3;



*****************************************************************************;
***                Call macro to fit nonlinear mixed model                ***;
***           (For details, see documentation for MIXTRAN macro)          ***;
*****************************************************************************;

%mixtran (data=datamrec, response=R, foodtype=fish, subject=subjectid,
          repeat=repeat, covars_prob=&covnlmixmod, covars_amt=&covnlmixmod,
          outlib=mylib, modeltype=corr, lambda=, replicate_var=replicvar,
          seq=, weekend=, vargroup=, numvargroups=,
          subgroup=, start_val1=, start_val2=, start_val3=,  
          vcontrol=, nloptions=qmax=61, titles=2, printlevel=2);   

*****************************************************************************;
*****************************************************************************;
*****************************************************************************;



************************************************;
*** Create input data set for indivint macro ***;
************************************************;

data parampred(keep = subjectid p_var_u1 a_var_u2 cov_u1u2 a_var_e a_lambda x1b1 x2b2);
  if (_n_ = 1) then set mylib._param_fish;
  set mylib._pred_fish;
run;


data parsubj1rec;
  merge parampred subj1rec(keep = subjectid R1 R2);
  by subjectid;
run;


data paramsubj1rec;
  if _n_=1 then set outmin_amt;
  set parsubj1rec;
  
  lamt = &boxcox_t_param;

run;



/*************************************************************************/
/*************************************************************************/
/*                                                                       */
/*  Description of the INDIVINT Macro                                    */
/*                                                                       */
/*************************************************************************/
/*                                                                       */
/* The INDIVINT macro calculates predicted values for regression         */
/* calibration using methods from Kipnis et al. (Biometrics, 2009) and   */
/* using results from an amount-only model or a two-part model fit using */
/* the MIXTRAN macro.  The INDIVINT macro performs adaptive Gaussian     */
/* quadrature to predict usual intake for each individual, and the macro */
/* allows the user to provide a Box-Cox transformation parameter in      */
/* order to calculate the predicted values on a transformed scale.  The  */
/* results from this macro are intended for use in a subsequent          */
/* regression model as discussed by Kipnis et al. (Biometrics, 2009).    */
/*                                                                       */
/* The syntax for calling the INDIVINT macro is:                         */
/*                                                                       */
/* %indivint(model12=, subj1recdata=, recid=, r24vars=, min_amt=,        */
/*           var_u1=, var_u2=, cov_u1u2=, var_e=, lambda=, xbeta1=,      */
/*           xbeta2=, boxcox_t_lamt=, lamt=, dencalc=, denopt=,          */
/*           u1nlmix=, u2nlmix=, titles=, notesprt=);                    */
/*                                                                       */
/* where the parameters are described as follows.                        */
/*                                                                       */
/*  "model12"            Specifies the type of model that was fit prior  */
/*                       to calling this macro.  A value of 1 indicates  */
/*                       that an amount-only model was fit, and a value  */
/*                       of 2 indicates that a two-part model was fit    */
/*                       where part 1 is the probability part of the     */
/*                       model and part 2 is the amount part of the      */
/*                       model.                                          */
/*                                                                       */
/*  "subj1recdata"       Specifies a data set with 1 record for each     */
/*                       individual.                                     */
/*                                                                       */
/*  "recid"              Specifies an identification (ID) variable that  */
/*                       uniquely identifies each individual's record.   */
/*                                                                       */
/*  "r24vars"            Specifies the 24-hour recall variables with     */
/*                       values that are either non-negative or a SAS    */
/*                       missing value if the 24-hour recall is not      */
/*                       available.  Variables must be space delimited   */
/*                       as illustrated in the following example:        */
/*                       "r24vars=r24hr1 r24hr2".                        */
/*                       Note for Advanced Users:  If all 24-hour recall */
/*                       values are missing for each subject, then the   */
/*                       denominator integration should not be           */
/*                       performed, so the "dencalc" macro parameter     */
/*                       should be specified as "dencalc=n".             */
/*                                                                       */
/*  "min_amt"            Specifies a variable that provides the minimum  */
/*                       intake amount.  This value may be selected as   */
/*                       the smallest value among the observed           */
/*                       consumption-day amounts.  Note that the         */
/*                       specified variable provides the same value for  */
/*                       each individual.  This value will be divided in */
/*                       half and used in the calculations for the       */
/*                       numerator integration.                          */
/*                                                                       */
/*  "var_u1"             Specifies a variable that provides the variance */
/*                       estimate for u1, the random effect from the     */
/*                       probability part of the model.  If a variable   */
/*                       is specified, then the macro will use its value */
/*                       as a diagonal entry of the covariance matrix    */
/*                       which is either a 1x1 matrix or a 2x2 matrix    */
/*                       depending on the number of random effects that  */
/*                       are in the model.                               */
/*                                                                       */
/*  "var_u2"             Specifies a variable that provides the variance */
/*                       estimate for u2, the random effect from the     */
/*                       amount part of the model or from an amount-only */
/*                       model.  If a variable is specified, then the    */
/*                       macro will use its value as a diagonal entry of */
/*                       the covariance matrix which is either a 1x1     */
/*                       matrix or a 2x2 matrix depending on the number  */
/*                       of random effects that are in the model.        */
/*                                                                       */
/*  "cov_u1u2"           Specifies a variable that provides the estimate */
/*                       of the covariance(u1, u2) from the two-part     */
/*                       model.  If the two-part model was an            */
/*                       uncorrelated model, then the specified variable */
/*                       should have a value of zero for every           */
/*                       individual's record.                            */
/*                                                                       */
/*  "var_e"              Specifies a variable that provides the variance */
/*                       estimate for e, the within-person error term    */
/*                       from the amount part of the model or from an    */
/*                       amount-only model.                              */
/*                                                                       */
/*  "lambda"             Specifies a variable that provides the estimate */
/*                       of the Box-Cox parameter, lambda, from the      */
/*                       amount part of the model or from an amount-only */
/*                       model.                                          */
/*                                                                       */
/*  "xbeta1"             Specifies a variable that provides the linear   */
/*                       predictor values calculated using the           */
/*                       covariates and estimates of the fixed effects   */
/*                       parameters from the probability part of the     */
/*                       model.                                          */
/*                                                                       */
/*  "xbeta2"             Specifies a variable that provides the linear   */
/*                       predictor values calculated using the           */
/*                       covariates and estimates of the fixed effects   */
/*                       parameters from the amount part of the model or */
/*                       from an amount-only model.                      */
/*                                                                       */
/*  "boxcox_t_lamt"      If "boxcox_t_lamt=y" or "boxcox_t_lamt=Y" then  */
/*                       individual usual intake will be predicted on a  */
/*                       transformed scale where the Box-Cox             */
/*                       transformation is used with the Box-Cox         */
/*                       parameter value provided by the "lamt" macro    */
/*                       parameter.  The default value for               */
/*                       "boxcox_t_lamt" is "n".                         */
/*                                                                       */
/*  "lamt"               Specifies a variable that provides the Box-Cox  */
/*                       parameter value when "boxcox_t_lamt=y" or       */
/*                       "boxcox_t_lamt=Y".  The macro does not allow    */
/*                       the Box-Cox parameter to be zero.               */
/*                                                                       */
/*  "dencalc"            By default, "dencalc=y" so the denominator      */
/*                       integration is performed.                       */
/*                       Note for Advanced Users:  If all 24-hour recall */
/*                       variables are missing for each subject, then    */
/*                       the denominator integration should not be       */
/*                       performed, so the "dencalc" option should be    */
/*                       specified as "dencalc=n".                       */
/*                                                                       */
/*  "denopt"             By default, "denopt=y" so the denominator       */
/*                       optimization is performed as part of the        */
/*                       denominator integration calculations.           */
/*                       Note for Advanced Users:  In some situations    */
/*                       the denominator optimization is redundant       */
/*                       because the empirical Bayes estimates of u1 and */
/*                       u2 are available from the model fitting         */
/*                       software; therefore, in these situations,       */
/*                       setting "denopt=n" or "denopt=N" allows the     */
/*                       macro to skip this optimization step and use    */
/*                       the variables provided by the "u1nlmix" and     */
/*                       "u2nlmix" macro parameters.                     */
/*                                                                       */
/*  "u1nlmix"            Specifies a variable for an Advanced Users      */
/*                       option.  For details, see the description for   */
/*                       the "denopt" macro parameter.                   */
/*                                                                       */
/*  "u2nlmix"            Specifies a variable for an Advanced Users      */
/*                       option.  For details, see the description for   */
/*                       the "denopt" macro parameter.                   */
/*                                                                       */
/*  "titles"             Specifies the number of title lines to be       */
/*                       reserved for the user's titles.  One additional */
/*                       title line is used by the macro.  The default   */
/*                       value is "0".                                   */
/*                                                                       */
/*  "notesprt"           If "notesprt=n" or "notesprt=N" then notes are  */
/*                       not printed to the SAS log.  The default value  */
/*                       for "notesprt" is "y".                          */
/*                                                                       */
/*************************************************************************/

*****************************************************************************************;
*****************************      Call indivint macro      *****************************;
*****************************************************************************************;

%indivint(model12=2, subj1recdata=paramsubj1rec, recid=subjectid, r24vars=R1 R2, min_amt=min_amt, 
          var_u1=p_var_u1, var_u2=a_var_u2, cov_u1u2=cov_u1u2, var_e=a_var_e, lambda=a_lambda, 
          xbeta1=x1b1, xbeta2=x2b2, boxcox_t_lamt=y, lamt=lamt, dencalc=y, denopt=y, u1nlmix=, 
          u2nlmix=, titles=2, notesprt=y);

********************************************************************************************************;
********************************************************************************************************;
********************************************************************************************************;


data subj1recres;
  merge subj1rec _resdata(keep = subjectid indusint);
  by subjectid;
run;  


proc reg data=subj1recres outest=regout;
  freq replicvar;
  model logtotmerc = indusint;
  title3 "Linear Regression Model:  Adjusted for Measurement Error";
run;
title3;
  
  
  
********************************;
*** T0 and T1 transformation ***;
********************************;
 
data diethealthout;
  set regout(rename = (indusint=rcbeta));

  lamt = &boxcox_t_param;

  t0boxcox = (&t0**lamt - 1)/lamt;
  t1boxcox = (&t1**lamt - 1)/lamt;
  
  tdiffrcbeta = (t1boxcox - t0boxcox)*rcbeta;

run;  
  
 
  
***************************;
*** Print final results ***;
***************************;

proc print data=diethealthout(keep = rcbeta lamt t0boxcox t1boxcox tdiffrcbeta);
  title3 "Regression Calibration Beta Is rcbeta, t0=0.1, t1=1.0, tdiffrcbeta=(t1boxcox - t0boxcox)*rcbeta";
run;
title3;



