/*********************************************************************/
/*********************************************************************/
/*                                                                   */
/*          THE LEGUME ALLOCATION MACRO FOR HEI-2005                 */
/*          (hei2005.beanspeas.allocation.macro.sas)                 */                              
/*                                                                   */
/*********************************************************************/
/*                     VERSION 1.1         03/14/2013                */
/*                                                                   */
/*                                                                   */
/* This legume allocation macro is to be used in conjunction with    */
/* the HEI-2005 scoring macro.                                       */
/*								         */
/* The purpose is to allocate legumes (beans and peas) into either   */
/* Meat and Beans or Total Vegetables and Dark Green and Orange	  */
/* Vegetables and Legumes.  Legume intake counts as Meat             */
/* and Beans until the standard is met, then the rest count 	  */
/* toward Total Vegetables and Dark Green and Orange Vegetables      */
/* and Legumes.                                                      */
/*								         */	
/* The resulting dataset, which is named by the user, contains the   */
/* same variables as the supplied dataset, and creates four new      */
/* variables.  They include:					         */
/*                                                                   */				    
/*   LEGTYPE = a character variable that represents if the legumes   */
/*       are added to meat ('ALLMEAT'), to vegetables ('ALLVEG')     */
/*       or distributed to both ('MEAT/VEG').  This variable is for  */
/*       informational purposes and is not required by the HEI-2005  */
/*       scoring macro.                                              */
/*                                                                   */
/*   Three additional variables that contain the total intake        */
/*    of Meat and Beans, Total Vegetables, and Dark Green Vegetables */
/*    and Legumes AFTER the appropriate amount of legumes has been   */
/*    added.  These new variables are named by appending             */
/*    'legume_added_' to the original name of each respective        */
/*     food group variable.                                          */                                                                  
/*								         */
/* The code has been created to work with variables from the 	 */
/* MyPyramid Equivalents Database but can be adapted to use          */
/* variables from other sources (e.g., data on the food supply).     */
/*								         */
/* The syntax for calling the macro is:                              */
/*                                                                   */
/* %LEGALLOC                                                         */
/* (indat=, kcal=, allmeat=, v_total=, v_dol=, legumes=, outdat=)    */ 
/*                                                                   */
/*  where                                                            */
/*                                                                   */
/*   "indat"        * Specifies the dataset to be used.              */
/*                                                                   */
/*   "kcal"         * Specifies calorie intake.                      */
/*                                                                   */
/*   "allmeat"      * Specifies sum of intake of 		         */
/*                      M_MPF+M_EGG+M_NUTSD+M_SOY.                   */
/*                                                                   */
/*   "v_total"      * Specifies the intake of V_TOTAL.               */
/*                                                                   */
/*   "v_dol"        * Specifies the intake of V_ORANGE+V_DRKGR.      */
/*                                                                   */
/*   "legumes"      * Specifies the intake of LEGUMES.               */
/*                                                                   */
/*   "outdat"       * Specifies the name of the resulting dataset.   */
/*                                                                   */
/*                                                                   */
/* Caution:  variable names "LEGTYPE", "MBMAX", "MEATLEG",           */
/*   "NEEDMEAT", "EXTRMEAT", "EXTRLEG" are reserved for this macro.  */
/*                                                                   */
/*                                                                   */
/*********************************************************************/
;



%macro LEGALLOC (indat=,kcal=,allmeat=,v_total=,v_dol=,legumes=,outdat=);

data &outdat (drop=mbmax meatleg needmeat extrmeat extrleg);
  set &indat;

/**Allocate legumes to either meat or veg**/
/*Standard is 2.5 oz equiv/1000 kcal*/

MBMAX=2.5*(&KCAL/1000);	/*Create the Meat and Beans standard*/

 /*Legumes intake calculation*/
 /*Legumes intake counts as Meat and Beans until the standard is met, then the rest count as Total Vegetables*/

/*(a) If total meat intake is less than the Meat and Beans recommendation then 
  all Legumes go to Meat and Beans*/
IF &ALLMEAT < MBMAX THEN DO;
   MEATLEG=&LEGUMES*4;	/*Convert cup equivalents of Legumes to oz equivalents */
   NEEDMEAT=MBMAX-&ALLMEAT;

   IF MEATLEG <= NEEDMEAT THEN DO;
      LEGTYPE='ALLMEAT ';
      legume_added_&ALLMEAT=&ALLMEAT+MEATLEG;
      legume_added_&V_TOTAL=&V_TOTAL;
      legume_added_&V_DOL=&V_DOL;
   END; 

/*(b) Some Legumes go to Meat and Beans and the rest go to Total Vegetables*/
   ELSE IF MEATLEG > NEEDMEAT THEN DO; 
      LEGTYPE='MEAT/VEG';
      EXTRMEAT=MEATLEG-NEEDMEAT;
      EXTRLEG=EXTRMEAT/4; 	/*Convert oz equivalents of Meat and Beans from Legumes back to cup equivalents*/
      legume_added_&ALLMEAT=&ALLMEAT+NEEDMEAT;
      legume_added_&V_TOTAL=&V_TOTAL+EXTRLEG;
      legume_added_&V_DOL=&V_DOL+EXTRLEG;
     END;
END; 

/*(c) If total meat intake exceeds the Meat and Beans standard, then all Legumes count as Total Vegetables*/
ELSE IF &ALLMEAT >= MBMAX THEN DO;  
    LEGTYPE='ALLVEG';
    legume_added_&ALLMEAT=&ALLMEAT;
    legume_added_&V_TOTAL=&V_TOTAL+&LEGUMES;
    legume_added_&V_DOL=&V_DOL+&LEGUMES;
END;

run;

%mend LEGALLOC;

 
/*  END OF THE LEGALLOC MACRO                                       */
/*******************************************************************/

