/*********************************************************************/
/*********************************************************************/
/*                                                                   */
/*                 THE HEI-2005 SCORING MACRO                        */                                               
/*                  (hei2005.score.macro.sas)                        */
/*********************************************************************/
/*                     VERSION 1.1         03/14/2013                */
/*                                                                   */
/*                                                                   */
/* This HEI-2005 macro is to be used to calculate densities and      */
/* and HEI-2005 component and total scores.                          */
/*								                     */
/* The macro requires an input dataset with variables for each of    */
/* the HEI-2005 components, noted below.                             */
/*											   */	
/* Users should first use the legumes (beans and peas) allocation	   */
/* macro to distribute legumes appropriately to the meat and         */
/* vegetable components. See http://riskfactor.cancer.gov/tools/hei/ */
/* for further details.								   */
/*											   */
/* The resulting dataset, which is named by the user, contains the   */
/* same variables as the supplied dataset, and creates 25 new        */
/* variables. These include:					               */
/*                                                                   */				    
/*   The densities (per 1000 kcal) or percent (of total calories)    */
/*    for each of the 12 HEI-2005 components.			         */
/*                                                                   */
/*   Scores for the 12 components of the HEI-2005.                   */
/*                                                                   */
/*   The total HEI-2005 score, which is the sum of the               */
/*    scores for the 12 components.                                  */
/*                                                                   */                                                                  
/* The syntax for calling the macro is:                              */
/*                                                                   */
/* %HEI2005                                                          */
/* (indat= ,kcal= ,f_total= ,wholefrt= ,lv_total= ,lv_dol= ,	   */
/*  g_total= ,g_whl= ,d_total=  ,lallmeat= ,oil= ,sfat= ,sodi= ,     */
/*  exfaas= ,outdat=); 						               */
/*                                                                   */
/*  where                                                            */
/*                                                                   */
/*   "indat"        * Specifies the dataset to be used.              */
/*                                                                   */
/*   "kcal"         * Specifies calorie amount.                      */
/*                                                                   */
/*   "f_fotal"      * Specifies the intake of total fruit.           */
/*                                                                   */
/*   "wholefrt"     * Specifies the intake of whole fruit.           */
/*                                                                   */
/*   "lv_total"     * Specifies the intake of total vegetables after */
/*                       legume allocation.                          */
/*                                                                   */
/*   "lv_dol"       * Specifies the intake of dark green and orange  */
/*                       vegetables and legumes after legume         */
/*				 allocation.                                 */
/*   "g_total"      * Specifies the intake of total grain.      */
/*                                                                   */
/*   "g_whl"        * Specifies the MPED intake of whole grain.      */
/*                                                                   */
/*   "d_fotal"      * Specifies the MPED intake of total dairy.      */
/*                                                                   */
/*   "lallmeat"     * Specifies the MPED intake of total meat        */
/*                       after legume allocation.                    */
/*                                                                   */
/*   "oil"          * Specifies the MPED grams of added fat in oil.  */
/*                                                                   */
/*   "sfat"         * Specifies the grams of saturated fat.          */
/*                                                                   */
/*   "sodi"         * Specifies the mg of sodium.                    */
/*                                                                   */                                                                  
/*   "exfaas"       * Specifies the total amount of extra calories.  */
/*                                                                   */
/*   "outdat"       * Specifies the name of the resulting dataset.   */
/*                                                                   */                                                                  
/* Caution:  variable names "SODMAX", "SODMIN", "SODMED",            */
/*   "SOFAMIN", "SOFAMAX" are reserved for this macro.               */
/*                                                                   */
/*                                                                   */
/*********************************************************************/
;



%macro HEI2005 (indat= ,kcal= ,f_total= ,wholefrt= ,lv_total= ,lv_dol= ,g_total= ,g_whl= ,d_total= ,
  lallmeat= ,oil= ,sfat= ,sodi= ,exfaas= ,outdat=);

data &outdat (drop=SODMAX SODMIN SODMED SOFAMIN SOFAMAX);
  set &indat;


/**Calculate HEI-2005 TOTAL FRUIT component score**/
/*Standard for maximum score is >=0.8 cup equiv/1000 kcal, Maximum score is 5;
  no Fruit intake, minimum score is zero*/
FRTDEN=&F_TOTAL/(&KCAL/1000);
HEI1_TOTALFRUIT=5*(FRTDEN/.8);
IF HEI1_TOTALFRUIT > 5 THEN HEI1_TOTALFRUIT=5;
IF &F_TOTAL=0 THEN HEI1_TOTALFRUIT=0;	

/**Calculate HEI-2005 WHOLE FRUIT component score**/
/*Standard for maximum score is >=0.4 cup equiv/1000 kcal, Maximum score is 5;
  no Fruit intake, minimum score is zero*/
WHFRDEN=&WHOLEFRT/(&KCAL/1000);
HEI2_WHOLEFRUIT=5*(WHFRDEN/.4); 
IF HEI2_WHOLEFRUIT > 5 THEN HEI2_WHOLEFRUIT=5;
IF &F_TOTAL=0 THEN HEI2_WHOLEFRUIT=0;	

/**Calculate HEI-2005 Total Vegetable component score**/
/*Standard for maximum score is >=1.1 cup equiv/1000 kcal, Maximum score is 5;
  no Vegetable intake, minimum score is zero*/
VEGDEN=&LV_TOTAL/(&KCAL/1000);
HEI3_TOTALVEG=5*(VEGDEN/1.1);
IF HEI3_TOTALVEG > 5 THEN HEI3_TOTALVEG=5;
IF &LV_TOTAL=0 THEN HEI3_TOTALVEG=0;

/**Calculate HEI-2005 Dark Green and Orange Vegetables and Legumes component score**/
/*Standard for maximum score is >=0.4 cup equiv/1000 kcal, Maximum score is 5*/
DGVDEN=&LV_DOL/(&KCAL/1000);
HEI4_DARKVEG=5*(DGVDEN/.4);
IF HEI4_DARKVEG > 5 THEN HEI4_DARKVEG=5;
IF &LV_DOL=0 THEN HEI4_DARKVEG=0;


/**Calculate HEI-2005 Total Grains component score**/
/*Standard for maximum score is >=3.0 cup equiv/1000 kcal, Maximum score is 5;
  no Total Grains intake, minimum score is zero*/
GRNDEN=&G_TOTAL/(&KCAL/1000);
HEI5_TOTALGRAIN=5*(GRNDEN/3.0);
IF HEI5_TOTALGRAIN > 5 THEN HEI5_TOTALGRAIN=5;

/**Calculate HEI-2005 Whole Grains component score**/
/*Standard for maximum score is >=1.5 cup equiv/1000 kcal, Maximum score is 5;
  no Whole Grains intake, minimum score is zero*/
WGRNDEN=&G_WHL/(&KCAL/1000);
HEI6_WHOLEGRAIN=5*(WGRNDEN/1.5);
IF HEI6_WHOLEGRAIN > 5 THEN HEI6_WHOLEGRAIN=5;

IF &G_TOTAL=0 THEN DO;  /*no reported Total Grains intake, Total Grains and Whole Grains component scores are zero*/
  HEI5_TOTALGRAIN=0; HEI6_WHOLEGRAIN=0;
END;

/**Calculate HEI-2005 Milk component score**/
/*Standard for maximum score is >=1.3 cup equiv/1000 kcal, Maximum score is 10;
  no Milk intake, minimum score is zero*/
DAIRYDEN=&D_TOTAL/(&KCAL/1000);
HEI7_MILK=10*(DAIRYDEN/1.3);
IF HEI7_MILK > 10 THEN HEI7_MILK=10;
IF &D_TOTAL=0 THEN HEI7_MILK=0;


       /*Calculate HEI-2005 Meat and Beans component score*/
MEATDEN=&LALLMEAT/(&KCAL/1000);
HEI8_MEATBEAN=10*(MEATDEN/2.5);
IF HEI8_MEATBEAN > 10 THEN HEI8_MEATBEAN=10;
IF &LALLMEAT=0 THEN HEI8_MEATBEAN=0;

/**Calculate HEI-2005 Oils component score**/
/*Standard for maximum score is >=12 grams/1000 kcal, Maximum score is 10*/
OILDEN=&OIL/(&KCAL/1000);
HEI9_OIL=10*(OILDEN/12);
IF HEI9_OIL > 10 THEN HEI9_OIL=10;
IF &OIL=0 THEN HEI9_OIL=0;

/**Calculate HEI-2005 Saturated Fat component score**/
/*Standard for maximum score is <=7% total kcal, Maximum score is 10;
 10% total kcal, score is 8; >=15% total kcal, minimum score is zero*/

IF &KCAL > 0 THEN PCTSFAT=100*(&SFAT*9)/&KCAL; /*Calculate percent of calories from Saturated Fat*/

IF PCTSFAT >= 15 THEN HEI10_SATFAT=0;
  ELSE IF PCTSFAT <= 7 THEN HEI10_SATFAT=10;
  ELSE IF PCTSFAT > 10 THEN 
      HEI10_SATFAT= 8 - ( 8 * (PCTSFAT-10)/5 );
  ELSE HEI10_SATFAT= 10 - (2 * (PCTSFAT-7)/3 );

/**Calculate HEI-2005 sodium component score**/
/*Standard for maximum score is <=0.7 grams/1000 kcal, Maximum score is 10;
  1.1 grams/1000 kcal, score is 8; >=2.0 grams/1000 kcal, minimum score is zero*/

IF &KCAL > 0 THEN SODDEN=&SODI/(&KCAL/1000); /*Calculate Sodium density (mg/1000 kcal); sodium intake in mg*/
SODMAX=2000;                                                                    
SODMED=1100;                                                                    
SODMIN=700;                                                                     
IF SODDEN >= SODMAX THEN HEI11_SODIUM=0;                                               
  ELSE IF SODDEN <= SODMIN THEN HEI11_SODIUM=10;                                       
  ELSE IF SODDEN >= SODMED THEN                                                 
      HEI11_SODIUM= 8 - ( 8 * (SODDEN-SODMED)/(SODMAX-SODMED) );                       
  ELSE HEI11_SODIUM= 10 - (2 * (SODDEN-SODMIN)/(SODMED-SODMIN) ); 

/**Calculate HEI-2005 Calories from SoFAAS component score**/
/*Standard for maximum score is <=20% total kcal, Maximum score is 20;
  >=50% total kcal, minimum score is zero*/


  /*Calculate SoFAAS as in %kcal
  Note: This results in more than 100% of total caloric intake from SoFAAS in a few cases 
  	due to the use of the general factors, 4 kcal/gram of carbohydrate, 9 kcal/gram of fat, 
  	and 7 kcal/gram of alcohol*/

IF &KCAL > 0 THEN SOFA_PERC=100*(&EXFAAS/&KCAL); 

SOFAMIN=20;
SOFAMAX=50;
IF SOFA_PERC >= SOFAMAX THEN HEI12_EXFAAS=0;
  ELSE IF SOFA_PERC <= SOFAMIN THEN HEI12_EXFAAS=20;
  ELSE HEI12_EXFAAS= 20 - ( 20* (SOFA_PERC-SOFAMIN) / (SOFAMAX-SOFAMIN) );

/*For individuals with no reported total energy intake for the day, 
  the HEI component and total scores are set to zero*/
IF &KCAL=0 THEN DO;
  HEI1_TOTALFRUIT=0; HEI2_WHOLEFRUIT=0; HEI3_TOTALVEG=0; HEI4_DARKVEG=0; HEI5_TOTALGRAIN=0; HEI6_WHOLEGRAIN=0; HEI7_MILK=0;
  HEI8_MEATBEAN=0; HEI9_OIL=0; HEI10_SATFAT=0; HEI11_SODIUM=0; HEI12_EXFAAS=0;
  END;

/**Calculate HEI-2005 total score**/
/*total HEI-2005 score is the sum of 12 HEI component scores*/
HEI2005_TOTAL_SCORE=HEI1_TOTALFRUIT+HEI2_WHOLEFRUIT+HEI3_TOTALVEG+HEI4_DARKVEG+HEI5_TOTALGRAIN+HEI6_WHOLEGRAIN+HEI7_MILK+HEI8_MEATBEAN+HEI9_OIL+HEI10_SATFAT+HEI11_SODIUM+HEI12_EXFAAS;

LABEL HEI2005_TOTAL_SCORE='TOTAL HEI-2005 SCORE'
          HEI1_TOTALFRUIT='HEI-2005 COMPONENT 1 TOTAL FRUIT'
	  HEI2_WHOLEFRUIT='HEI-2005 COMPONENT 2 WHOLE FRUIT'
	  HEI3_TOTALVEG='HEI-2005 COMPONENT 3 TOTAL VEGETABLES'
	  HEI4_DARKVEG='HEI-2005 COMPONENT 4 DARK GREEN & ORANGE VEG & LEGUMES'
	  HEI5_TOTALGRAIN='HEI-2005 COMPONENT 5 TOTAL GRAINS'
	  HEI6_WHOLEGRAIN='HEI-2005 COMPONENT 6 WHOLE GRAINS'
	  HEI7_MILK='HEI-2005 COMPONENT 7 MILK'
	  HEI8_MEATBEAN='HEI-2005 COMPONENT 8 MEAT & BEANS'
	  HEI9_OIL='HEI-2005 COMPONENT 9 OILS'
	  HEI10_SATFAT='HEI-2005 COMPONENT 10 SATURATED FAT'
	  HEI11_SODIUM='HEI-2005 COMPONENT 11 SODIUM'
	  HEI12_EXFAAS='HEI-2005 COMPONENT 12 CALORIES FROM SOLID FAT, ALCOHOL & ADDED SUGAR (SoFAAS)'
	  FRTDEN='DENSITY OF MPED TOTAL FRUIT PER 1000 KCAL'
	  WHFRDEN='DENSITY OF MPED WHOLE FRUIT PER 1000 KCAL'
	  VEGDEN='DENSITY OF MPED TOTAL VEGETABLES PER 1000 KCAL'
	  DGVDEN='DENSITY OF MPED DARK GREEN AND ORANGE VEGETABLES AND LEGUMES PER 1000 KCAL'
	  GRNDEN='DENSITY OF MPED TOTAL GRAIN PER 1000 KCAL'
	  WGRNDEN='DENSITY OF MPED OF WHOLE GRAIN PER 1000 KCAL'
	  DAIRYDEN='DENSITY OF MPED OF DAIRY PER 1000 KCAL'
	  MEATDEN='DENSITY OF MPED TOTAL MEAT/PROTEIN PER 1000 KCAL'
	  OILDEN='DENSTIY OF GRAMS DISCRETIONARY FAT IN OIL PER 1000 KCAL'
	  PCTSFAT='PERCENT OF CALORIES FROM SATURATED FAT'
	  SODDEN='DENSITY OF SODIUM PER 1000 KCAL'
	  SOFA_PERC='PERCENT OF CALORIES FROM ADDED SUGAR, SOLID FAT, AND ALCOHOL'
	  ;
run;


%mend HEI2005;


/*  END OF THE HEI-2005 SCORING MACRO                               */
/*******************************************************************/




