
downloadAcclRawData80Hz <- function(destDir, data="NNYFS", subject.seqn="72102", extract=TRUE, 
                                compress=TRUE, zipcmd=NULL, checkForFiles=TRUE,
                                delete.bz2=TRUE, DEBUG=FALSE) {

  destDir <- check_destDir(destDir)
  data    <- check_data(data)
  zipcmd  <- check_zipcmd(zipcmd)
  op      <- list(extract=extract, compress=compress, subject.seqn=subject.seqn,
                  download.options=NULL, checkForFiles=checkForFiles, delete.bz2=delete.bz2,
                  zipcmd=zipcmd, DEBUG=DEBUG)
  
  ret     <- downloadRaw_main(destDir, data, op)
 
  ret
}

getSEQNids <- function(data) {

  data <- check_data(data)
  tmp  <- getRawAccDataURLs()
  url  <- tmp[data]
  tmp  <- getFileNamesFromUrl(url)
  ret  <- tmp$SEQN
  ret

}

getDataFileInfo <- function(data, type=NULL) {

  data <- check_data(data)
  type <- check_type(type, data)
  ret  <- dataFileInfo(data, type=type)
  
  ret

}

downloadFiles <- function(destDir, data, files=NULL, save.type="csv") {

  destDir <- check_destDir(destDir)
  data    <- check_data(data)
  info    <- check_files(files, data)
  ext     <- check_save.type(save.type)
  n       <- nrow(info)
  fcol    <- infoFileCol()
  pcol    <- infoPathCol()

  for (i in 1:n) {
    destfile <- paste0(destDir, data, "_", info[i, fcol, drop=TRUE], ".xpt")
    try(my_download.file(info[i, pcol], destfile, NULL), silent=TRUE)
    if (!file.exists(destfile)) {
      warning(paste0("Problem downloading file ", files[i]))
      next
    }
    if (ext != "xpt") {
      savefile <- paste0(destDir, data, "_", info[i, fcol, drop=TRUE], ".", ext)
      saveXptFileAs(destfile, savefile, ext=ext)
      file.remove(destfile)
    } 
  }
  NULL
}

downloadAndReadFile <- function(data, file, delete=TRUE) {

  data     <- check_data(data)
  info     <- check_files(file, data, name="file", len=1)
  destDir  <- check_destDir(getwd(), name="getwd()")
  destfile <- paste0(destDir, data, "_", info[1, infoFileCol(), drop=TRUE], ".xpt")
  my_download.file(info[1, infoPathCol()], destfile, NULL)
  if (!file.exists(destfile)) {
    msg <- paste0("ERROR: downloading file ", file, " to ", destfile)
    stop(msg)
  }
  ret <- read_file(destfile)
  if (delete) file.remove(destfile)
  ret   
}

merge2DataObjects <- function(obj1, obj2) {

  obj1 <- check_data_object(obj1, "obj1")
  obj2 <- check_data_object(obj2, "obj2")
  ret  <- merge2(obj1, obj2)
  ret

}

dataFileInfo <- function(data, type=NULL) {

  if (data == name_nnyfs()) {
    ret <- dataFileInfo_NNYFS_2012(type) 
  } else if (data == name_2013_14()) {
    ret <- dataFileInfo_NHANES_2013_2014(type)
  } else if (data == name_2011_12()) {
    ret <- dataFileInfo_NHANES_2011_2012(type)
  }
  ret

}

downloadRaw_main <- function(destDir, data, op) {

  DEBUG <- op$DEBUG

  # Create new folder
  dir <- paste0(destDir, data, "/")
  if (!dir.exists(dir)) dir.create(dir) 
  if (!dir.exists(dir)) stop(paste0("ERROR: trying to create ", dir))

  # Get the url
  urls <- getRawAccDataURLs()
  url  <- urls[data]
  miss <- NULL
  
  # Get the files to download, option subject.seqn takes precedance
  tmp       <- getFileNamesFromUrl(url)
  all.files <- tmp$files
  all.seqn  <- tmp$SEQN 
  tmp       <- check_subject.seqn(op[["subject.seqn", exact=TRUE]], all.seqn, name="subject.seqn")  
  if (length(tmp)) {
    seqn  <- tmp[["seqn", exact=TRUE]]
    miss  <- tmp[["missing", exact=TRUE]]
    tmp   <- all.seqn %in% seqn
    seqn  <- all.seqn[tmp]
    files <- all.files[tmp]
  } else {
    nfiles <- length(all.seqn)
    seqn   <- all.seqn[1:nfiles]
    files  <- all.files[1:nfiles] 
    size   <- getRawDataSize(data)
    message("**************************************************")
    msg    <- paste0("NOTE: Raw data for all subjects will be downloaded, total size is ", size, " (compressed).")
    message(msg)
    message("**************************************************")
    Sys.sleep(3)
  }
  nfiles    <- length(files)
  if (!nfiles) stop("ERROR: all specified subjects do not have accelerometer data")
  destfiles <- paste0(dir, basename(files))

  tmp              <- list(method="auto", quiet=FALSE, mode="w", cacheOK=TRUE,
                           extra=getOption("download.file.extra"), headers=NULL)
  download.options <- default.list(op[["download.options", exact=TRUE]], names(tmp), tmp)

  # Set env variable if needed
  if (OSisWindows()) rc <- setPathEnvVar_windows(op$zipcmd, DEBUG=DEBUG)

  extract       <- op$extract
  compress      <- op$compress
  checkForFiles <- op$checkForFiles
  delete.bz2    <- op$delete.bz2
  err.download  <- NULL
  err.extract   <- NULL
  
  for (i in 1:nfiles) {
    seqid    <- seqn[i]
    bz2file  <- destfiles[i]
    download <- 1
    if (checkForFiles && file.exists(bz2file)) download <- 0
    if (download) {
      message(paste0("Downloading file ", i, " of ", nfiles))
      rc <- my_download.file(files[i], bz2file, download.options) 
      if (rc) {
        warning(paste0("Bad return code when downloading ", files[i]))
        file.remove(bz2file)
        err.download <- c(err.download, seqid)  
        next
      }
    } 

    if (extract) {
      # Create folder to extract results
      dir2 <- paste0(dir, seqid, "/") 
      if (!dir.exists(dir2)) dir.create(dir2) 
      if (!dir.exists(dir2)) {
        warning(paste0("ERROR creating ", dir2))
        next
      }
      rc <- checkRawDataDirToExtract(dir2, bz2file, checkForFiles)
      if (rc) {
        message("Extracting files")
        extractFilesFromBz2(bz2file, dir2, zipcmd=op$zipcmd, DEBUG=DEBUG)
      }
      if (compress) {
        try(compressFiles(files=NULL, dir=dir2, with.substr=".sensor.csv", zipcmd=op$zipcmd, DEBUG=DEBUG))
      }
    }
    if (delete.bz2 && file.exists(bz2file)) file.remove(bz2file)  
  }
  list(missing.data=miss, error.download=err.download)
}

checkRawDataDirToExtract <- function(dir, bz2File, checkForFiles) {

  ret <- 1 # Assume data must be extracted
  ff  <- list.files(dir, full.names=FALSE)
  if (!length(ff)) return(ret)

  # Files exist in the folder, so get all the files that are supposed to be in the folder
  bz2files    <- getBaseFilesFromBz2(bz2File)
  if (!length(bz2files)) {
    bz2files  <- paste0(dir, ff)
  } else {
    bz2files  <- paste0(dir, bz2files)
  }
  bz2files.gz <- paste0(bz2files, ".gz")
  
  rem <- 1
  if (checkForFiles) {
    # See if all the files are there. If not, then delete and re-extract. To do: make this better
    tmp <- file.exists(bz2files) | file.exists(bz2files.gz)
    if (all(tmp)) {
      ret <- 0 # No need to extract
      rem <- 0 # Do not delete files
    } 
  }

  if (rem) {
    removeFiles(c(bz2files, bz2files.gz))
  }

  ret
}

getRawAccDataURLs <- function() {

  pax_g      <- rawPath_2011_12()
  pax_h      <- rawPath_2013_14()
  pax_y      <- rawPath_nnyfs()
  ret        <- c(pax_g, pax_h, pax_y)
  names(ret) <- c(name_2011_12(), name_2013_14(), name_nnyfs())
  
  ret
}

getRawDataSize <- function(data) {

  ret        <- c(rawSize_2013_14(), rawSize_2011_12(), rawSize_nnyfs())
  names(ret) <- c(name_2013_14(), name_2011_12(), name_nnyfs())
  if (length(data)) ret <- ret[data]
  ret

}

getFileNamesFromUrl <- function(url) {

  str  <- ".tar.bz2"
  x    <- scan(url, what="char", quiet=TRUE)
  tmp  <- grepl(str, x, fixed=TRUE)
  x    <- x[tmp]
  if (!length(x)) stop("ERROR 1 in getFileNamesFromUrl")
  x    <- parseDelimVec0(x, str, 3, numeric=0)
  if (ncol(x) != 3) stop("ERROR 2 in getFileNamesFromUrl")
  x    <- x[, 2, drop=TRUE]
  x    <- gsub(">", "", x, fixed=TRUE)
  x    <- gsub('\"', '', x, fixed=TRUE)
  x    <- gsub(str, "", x, fixed=TRUE)
  numx <- as.numeric(x)
  tmp  <- !is.finite(numx)
  if (any(tmp)) {
    print(x[tmp])
    stop("ERROR 3 in getFileNamesFromUrl")
  } 
  ff  <- paste0(url, x, str)
  ret <- list(files=ff, SEQN=numx)
  ret
}

my_download.file <- function(url, destfile, op) {

  if (OSisWindows()) {
    rc <- download.file(url, destfile, mode="wb")
  } else {
    rc <- download.file(url, destfile)
  }

  rc
}

getInfoFromFileName <- function(f) {

  #GT3XPLUS-AccelerationCalibrated-[FIRMWARE_VERSION].[SENSOR_ID].[YYYY]-[MM]-[DD]-[HH]-[MM]-[SS]-000-P0000.sensor.csv.gz

  bf      <- basename(f)
  remstrs <- c("GT3XPLUS-AccelerationCalibrated-", "-000-P0000.sensor.csv")
  for (str in remstrs) bf <- gsub(str, "", bf, fixed=TRUE)
  vec <- getVecFromStr(bf, delimiter=".")
  len <- length(vec)
  if (len != 3) stop("ERROR 1 in getInfoFromFileName")
  firmware_version <- vec[1]
  sensor_id        <- vec[2]
  timestamp        <- vec[3]
  vec              <- getVecFromStr(timestamp, delimiter="-")
  len              <- length(vec)
  if (len != 6) stop("ERROR 2 in getInfoFromFileName")
  year   <- vec[1]
  month  <- vec[2]
  day    <- vec[3]
  hour   <- vec[4]
  minute <- vec[5]
  second <- vec[6] 
 
  list(firmware_version=firmware_version, sensor_id=sensor_id, timestamp=timestamp,
       year=year, month=month, day=day, hour=hour, minute=minute, second=second)

}

setDataFileInfo <- function(x, path) {

  x                  <- matrix(data=x, ncol=4, byrow=TRUE)
  colnames(x)        <- c(infoNameCol(), infoFileCol(),infoSizeCol(), infoPubCol())
  x                  <- as.data.frame(x, stringsAsFactors=FALSE)
  x[, infoPathCol()] <- paste0(path, x[, infoFileCol(), drop=TRUE], ".XPT")
  x

}

updateDataFileInfo <- function(base, new, new.type) {
  
 new <- data.frame(Type=new.type, new)
 if (!length(base)) return(new)
 
 cx1 <- colnames(base)
 cx2 <- colnames(new)
 if (!all(cx1 == cx2)) stop("ERROR 1")
 ret <- rbind(base, new)
 ret 
 
}

dataFileInfo_NHANES_2011_2012 <- function(type) {
  
  base <- NULL
  nms  <- names(type)
  
  if ("DE" %in% type) {
    new  <- dataFileInfo_2011_demo() 
    base <- updateDataFileInfo(base, new, nms[type=="DE"])
  }
  if ("DI" %in% type) {
    new  <- dataFileInfo_2011_diet() 
    base <- updateDataFileInfo(base, new, nms[type=="DI"])
  }
  if ("EX" %in% type) {
    new  <- dataFileInfo_2011_exam() 
    base <- updateDataFileInfo(base, new, nms[type=="EX"])
  }
  if ("LA" %in% type) {
    new  <- dataFileInfo_2011_lab() 
    base <- updateDataFileInfo(base, new, nms[type=="LA"])
  }
  if ("QU" %in% type) {
    new  <- dataFileInfo_2011_question() 
    base <- updateDataFileInfo(base, new, nms[type=="QU"])
  }
  
  base
  
}

dataFileInfo_2011_demo <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2011-2012/"
  
  x <- c(
      "Demographic Variables & Sample Weights", "DEMO_G",  "3.6 MB", "January 2015"
       )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2011_diet <- function() {
  
  path  <- "https://wwwn.cdc.gov/Nchs/Nhanes/2011-2012/"
  path2 <- "https://wwwn.cdc.gov/Nchs/Nhanes/1999-2000/"
  
  x <- c(
    "Dietary Interview - Individual Foods, First Day", "DR1IFF_G", "82 MB", "September 2014", 
    "Dietary Interview - Individual Foods, Second Day", "DR2IFF_G", "72.7 MB", "September 2014",
    "Dietary Interview - Total Nutrient Intakes, First Day", "DR1TOT_G", "11.8 MB", "September 2014",
    "Dietary Interview - Total Nutrient Intakes, Second Day", "DR2TOT_G", "5.9 MB", "September 2014",
    "Dietary Interview Technical Support File - Food Codes", "DRXFCD_G", "2.1 MB", "September 2014",
    "Dietary Interview Technical Support File - Modification Codes", "DRXMCD_G", "233.6 KB", "September 2014",
    "Dietary Supplement Use 24-Hour - Individual Dietary Supplements, First Day", "DS1IDS_G", "3 MB", "December 2014",
    "Dietary Supplement Use 24-Hour - Individual Dietary Supplements, Second Day", "DS2IDS_G", "3.1 MB", "December 2014",
    "Dietary Supplement Use 24-Hour - Total Dietary Supplements, First Day", "DS1TOT_G", "3.5 MB", "December 2014",
    "Dietary Supplement Use 24-Hour - Total Dietary Supplements, Second Day", "DS2TOT_G", "3.5 MB", "December 2014",
    "Dietary Supplement Use 30 Day - Individual Dietary Supplements", "DSQIDS_G", "7 MB", "December 2014",
    "Dietary Supplement Use 30-Day - Total Dietary Supplements", "DSQTOT_G", "2.9 MB", "December 2014"
  )
  ret <- setDataFileInfo(x, path)
  
  x <- c(
  "Dietary Supplement Database - Blend Information", "DSBI", "8.8 MB", "January 2022", 
  "Dietary Supplement Database - Ingredient Information", "DSII", "74 MB", "January 2022",
  "Dietary Supplement Database - Product Information", "DSPI", "5.7 MB", "January 2022"
  )
  ret2 <- setDataFileInfo(x, path2)
  
  rbind(ret, ret2)
  
}

dataFileInfo_2011_exam <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2011-2012/"
  
  x <- c(
    "Audiometry",                                                     "AUX_G",    "2.4 MB",   "April 2016",
    "Audiometry - Acoustic Reflex",                                   "AUXAR_G",  "11.6 MB",  "April 2016",
    "Audiometry - Tympanometry",                                      "AUXTYM_G", "5.8 MB",   "April 2016",
    "Blood Pressure",                                                 "BPX_G",    "1.9 MB",   "September 2013",
    "Body Measures",                                                  "BMX_G",    "1.9 MB",   "September 2015",
    "Dual-Energy X-ray Absorptiometry - Android/Gynoid Measurements", "DXXAG_G",  "1 MB",     "October 2021",
    "Dual-Energy X-ray Absorptiometry - Whole Body",                  "DXX_G",    "4.1 MB",   "March 2020",
    "Exhaled Nitric Oxide",                                           "ENX_G",    "1.3 MB",   "February 2014",
    "Fluorosis - Clinical",                                           "FLXCLN_G", "571.6 KB", "April 2019",
    "Muscle Strength - Grip Test",                                    "MGX_G",    "1.7 MB",   "December 2014",
    "Oral Health - Dentition",                                        "OHXDEN_G", "6.6 MB",   "April 2019",
    "Oral Health - Periodontal",                                      "OHXPER_G", "17 MB",    "March 2014",
    "Oral Health - Recommendation of Care",                           "OHXREF_G", "842 KB",   "March 2014",
    "Physical Activity Monitor - Day",                                "PAXDAY_G", "6.2 MB",   "November 2020",
    "Physical Activity Monitor - Header",                             "PAXHD_G",  "376.2 KB", "November 2020",
    "Physical Activity Monitor - Hour",                               "PAXHR_G",  "122.2 MB", "November 2020",
    "Physical Activity Monitor - Minute",                             "PAXMIN_G", "7.6 GB",   "November 2020",
    #"Physical Activity Monitor - Raw data",                           "FTP",      "1.04 TB",  "January 2022",
    "Spirometry - Pre and Post-Bronchodilator",                       "SPX_G",    "2.6 MB",   "December 2014",
    "Spirometry - Raw Curve Data",                                    "SPXRAW_G", "12.3 MB",  "December 2014",
    "Tuberculosis",                                                   "TBX_G",    "307 KB",   "January 2015")
  
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2011_lab <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2011-2012/"
  
  x <- c(
    "Albumin & Creatinine - Urine", "ALB_CR_G", "368.2 KB", "September 2016",
    "Apolipoprotein B", "APOB_G", "102.5 KB", "January 2014",
    "Arsenics - Total & Speciated - Urine", "UAS_G", "388.4 KB", "October 2014",
    "Arsenics - Total & Speciated - Urine - Special Sample", "UASS_G", "352.1 KB", "September 2013",
    "Brominated Flame Retardants (BFRs) - Pooled Samples", "BFRPOL_G", "88.9 KB", "November 2020",
    "Cadmium, Lead, Total Mercury, Selenium, & Manganese - Blood", "PBCD_G", "1.1 MB", "September 2013",
    "Caffeine & Caffeine Metabolites - Urine", "CAFE_G", "631.3 KB", "January 2020",
    "Chlamydia - Urine", "CHLMDA_G", "35 KB", "November 2013",
    "Cholesterol - HDL", "HDL_G", "184.5 KB", "September 2013",
    "Cholesterol - LDL & Triglycerides", "TRIGLY_G", "153.4 KB", "January 2014",
    "Cholesterol - Total", "TCHOL_G", "184.5 KB", "September 2013",
    "Complete Blood Count with 5-part Differential - Whole Blood", "CBC_G", "1.4 MB", "January 2014",
    "Copper, Selenium & Zinc - Serum", "CUSEZN_G", "184.4 KB", "October 2014",
    "Cotinine - Serum & Total NNAL - Urine", "COTNAL_G", "397 KB", "December 2013",
    "Cytomegalovirus IgG & IgM Antibodies - Serum", "CMV_G", "36.7 KB", "October 2013",
    "DEET and Metabolites", "DEET_G", "178.1 KB", "March 2017",
    "Environmental Phenols & Parabens", "EPH_G", "347.6 KB", "October 2014",
    "Fasting Questionnaire", "FASTQX_G", "1.3 MB", "September 2013",
    "Fatty Acids - Serum", "FAS_G", "2.7 MB", "February 2020",
    "Folate - RBC", "FOLATE_G", "211.1 KB", "February 2019",
    "Folate Forms - Total & Individual - Serum", "FOLFMS_G", "1 MB", "February 2019",
    "Glycohemoglobin", "GHB_G", "103.4 KB", "September 2013",
    "Hepatitis A Antibody", "HEPA_G", "137.6 KB", "September 2013",
    "Hepatitis B Surface Antibody", "HEPB_S_G", "137.6 KB", "September 2013",
    "Hepatitis B: Core Antibody, Surface Antigen; Hepatitis D Antibody", "HEPBD_G", "245.7 KB", "September 2013",
    "Hepatitis C: Confirmed Antibody, RNA (HCV-RNA), & Genotype", "HEPC_G", "245.7 KB", "February 2015",
    "Hepatitis E : IgG & IgM Antibodies", "HEPE_G", "184.5 KB", "September 2013",
    "Herpes Simplex Virus Type-1 & Type-2", "HSV_G", "86.9 KB", "November 2013",
    "HIV Antibody Test", "HIV_G", "62.4 KB", "September 2013",
    "Human Papillomavirus (HPV) - Oral Rinse", "ORHPV_G", "1.5 MB", "February 2014",
    "Human Papillomavirus (HPV) DNA - Vaginal Swab: Digene Hybrid Capture & Roche Linear Array", "HPVSWR_G", "661.1 KB", "March 2015",
    "Iodine - Urine", "UIO_G", "82.3 KB", "October 2014",
    "Mercury: Inorganic - Urine", "UHG_G", "102.7 KB", "October 2014",
    "Mercury: Inorganic, Ethyl and Methyl - Blood", "IHGEM_G", "561.6 KB", "September 2013",
    "Metals - Urine", "UHM_G", "592.4 KB", "October 2014",
    "Metals - Urine - Special Sample", "UHMS_G", "537 KB", "September 2013",
    "Methylmalonic Acid", "MMA_G", "125.9 KB", "October 2018",
    "Non-dioxin-like Polychlorinated Biphenyls & Mono-ortho-substituted Polychlorinated Biphenyls - Pooled Samples", "PCBPOL_G", "164.4 KB", "November 2020",
    "Oral Glucose Tolerance Test", "OGTT_G", "266.2 KB", "January 2014",
    "Organophosphate Insecticides - Dialkyl Phosphate Metabolites - Urine", "OPD_G", "276.6 KB", "November 2019",
    "Osmolality - Urine", "UCOSMO_G", "123.3 KB", "September 2013",
    "Perchlorate, Nitrate & Thiocyanate - Urine", "PERNT_G", "184.4 KB", "October 2014",
    "Perchlorate, Nitrate & Thiocyanate - Urine - Special Sample", "PERNTS_G", "167.2 KB", "September 2013",
    "Pesticides - Environmental - Urine", "PP_G",  "143.6 KB", "October 2014",
    "Pesticides - Organochlorine Pesticides - Pooled Samples", "PSTPOL_G", "63.7 KB", "November 2020",
    "Phthalates & Plasticizers Metabolites - Urine", "PHTHTE_G", "633.3 KB", "October 2014",
    "Plasma Fasting Glucose & Insulin", "GLU_G", "204.3 KB", "August 2016",
    "Polyaromatic Hydrocarbons (PAHs) - Urine", "PAH_G", "470.1 KB", "October 2014",
    "Polyaromatic Hydrocarbons (PAHs)- Urine - Special Sample", "PAHS_G", "426 KB", "February 2014",
    "Polyfluoroalkyl Chemicals", "PFC_G", "446.6 KB", "October 2014",
    "Pooled-Sample Technical Support File", "POOLTF_G", "51.5 KB", "April 2018",
    "Pregnancy Test - Urine", "UCPREG_G", "19.3 KB", "October 2014",
    "Pyrethroids, Herbicides, & Organophosphorus Metabolites - Urine", "UPHOPM_G", "276.6 KB", "September 2019",
    "Standard Biochemistry Profile", "BIOPRO_G", "1.9 MB", "February 2014",
    "Thyroid Profile", "THYROD_G", "240.8 KB", "October 2014",
    "Tissue Transglutaminase Assay (IgA-TTG) & IgA Endomyseal Antibody Assay (IgA EMA)", "TGEMA_G", "184.5 KB", "September 2013",
    "Total Testosterone", "TST_G", "123.3 KB", "March 2014",
    "Tuberculosis - Quantiferon_In_Gold", "TB_G", "307 KB", "November 2013",
    "Urine Flow Rate", "UCFLOW_G", "613.1 KB", "April 2018",
    "Vitamin B12", "VITB12_G", "125.9 KB", "March 2014",
    "Vitamin D", "VID_G", "631.7 KB", "October 2017",
    "Volatile Organic Compound (VOC) Metabolites - Urine", "UVOC_G", "1.1 MB", "November 2021",
    "Volatile Organic Compound (VOC) Metabolites - Urine - Special Sample", "UVOCS_G", "1 MB", "November 2021",
    "Volatile Organic Compounds (VOCs) - Blood", "VOCWB_G", "1.7 MB", "April 2017",
    "Volatile Organic Compounds (VOCs) - Trihalomethanes/MTBE/Nitromethane - Blood", "VOCMWB_G", "363 KB", "February 2018"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2011_question <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2011-2012/"
  
  x <- c(
    "Acculturation", "ACQ_G", "321.9 KB", "August 2014",
    "Alcohol Use", "ALQ_G", "440.8 KB", "October 2013",
    "Audiometry", "AUQ_G", "2.1 MB", "December 2013",
    "Blood Pressure & Cholesterol", "BPQ_G", "726.5 KB", "September 2013",
    "Cardiovascular Health", "CDQ_G", "481.6 KB", "September 2013",
    "Cognitive Functioning", "CFQ_G", "253.8 KB", "March 2017",
    "Consumer Behavior", "CBQ_G", "458.9 KB", "February 2015",
    "Creatine Kinase", "CKQ_G", "411.2 KB", "September 2013",
    "Current Health Status", "HSQ_G", "982.3 KB", "September 2013",
    "Dermatology", "DEQ_G", "267 KB", "August 2017",
    "Diabetes", "DIQ_G", "3.8 MB", "September 2013",
    "Diet Behavior & Nutrition", "DBQ_G", "3.2 MB", "February 2015",
    "Drug Use", "DUQ_G", "1.5 MB", "November 2013",
    "Early Childhood", "ECQ_G", "281.9 KB", "September 2013",
    "Food Security", "FSQ_G", "2.9 MB", "July 2015",
    "Health Insurance", "HIQ_G", "1.3 MB", "September 2013",
    "Hepatitis C Follow Up", "HCQ_G", "33.6 KB", "July 2014",
    "Hospital Utilization & Access to Care", "HUQ_G", "764.3 KB", "September 2013",
    "Housing Characteristics", "HOQ_G", "229.8 KB", "September 2013",
    "Immunization", "IMQ_G", "611.6 KB", "September 2013",
    "Income", "INQ_G", "1.1 MB", "February 2015",
    "Kidney Conditions - Urology", "KIQ_U_G", "697.9 KB", "November 2014",
    "Medical Conditions", "MCQ_G", "6.6 MB", "September 2013",
    "Mental Health - Depression Screener", "DPQ_G", "484.8 KB", "September 2013",
    "Occupation", "OCQ_G", "1 MB", "September 2017",
    "Oral Health", "OHQ_G", "2.4 MB", "March 2014",
    "Pesticide Use", "PUQMEC_G", "184.5 KB", "September 2013",
    "Physical Activity", "PAQ_G", "1.5 MB", "March 2017",
    "Physical Functioning", "PFQ_G", "2.2 MB", "September 2013",
    "Prescription Medications", "RXQ_RX_G", "2.9 MB", "July 2014",
    "Prescription Medications - Drug Information", "RXQ_DRUG", "2.6 KB", "September 2021",
    "Preventive Aspirin Use", "RXQASA_G", "227 KB", "September 2013",
    "Reproductive Health", "RHQ_G", "1.5 MB", "December 2017",
    "Respiratory Health", "RDQ_G", "1.1 MB", "September 2013",
    "Sexual Behavior", "SXQ_G", "1.8 MB", "September 2013",
    "Sleep Disorders", "SLQ_G", "194.2 KB", "November 2013",
    "Smoking - Cigarette Use", "SMQ_G", "1.9 MB", "May 2015",
    "Smoking - Household Smokers", "SMQFAM_G", "382.5 KB", "September 2013",
    "Smoking - Recent Tobacco Use", "SMQRTU_G", "1.2 MB", "September 2013",
    "Taste & Smell", "CSQ_G", "990.8 KB", "September 2013",
    "Tuberculosis", "TBQ_G", "831.4 KB", "November 2013",
    "Volatile Toxicant (Subsample)", "VTQ_G", "569.9 KB", "April 2017",
    "Weight History", "WHQ_G", "3 MB", "July 2014",
    "Weight History - Youth", "WHQMEC_G", "289.3 KB", "March 2014"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_NHANES_2013_2014 <- function(type) {
  
  base <- NULL
  nms  <- names(type)
  
  if ("DE" %in% type) {
    new  <- dataFileInfo_2013_demo() 
    base <- updateDataFileInfo(base, new, nms[type=="DE"])
  }
  if ("DI" %in% type) {
    new  <- dataFileInfo_2013_diet() 
    base <- updateDataFileInfo(base, new, nms[type=="DI"])
  }
  if ("EX" %in% type) {
    new  <- dataFileInfo_2013_exam() 
    base <- updateDataFileInfo(base, new, nms[type=="EX"])
  }
  if ("LA" %in% type) {
    new  <- dataFileInfo_2013_lab() 
    base <- updateDataFileInfo(base, new, nms[type=="LA"])
  }
  if ("QU" %in% type) {
    new  <- dataFileInfo_2013_question() 
    base <- updateDataFileInfo(base, new, nms[type=="QU"])
  }
  
  base
  
}

dataFileInfo_2013_demo <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2013-2014/"
  
  x <- c(
    "Demographic Variables & Sample Weights", "DEMO_H",  "3.7 MB", "October 2015"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2013_diet <- function() {
  
  path  <- "https://wwwn.cdc.gov/Nchs/Nhanes/2013-2014/"
  path2 <- "https://wwwn.cdc.gov/Nchs/Nhanes/1999-2000/"
  
  x <- c(
    "Dietary Interview - Individual Foods, First Day", "DR1IFF_H", "84.2 MB", "September 2016", 
    "Dietary Interview - Individual Foods, Second Day", "DR2IFF_H", "72.2 MB", "September 2016",
    "Dietary Interview - Total Nutrient Intakes, First Day", "DR1TOT_H", "12.6 MB", "September 2016",
    "Dietary Interview - Total Nutrient Intakes, Second Day", "DR2TOT_H", "6.4 MB", "September 2016",
    "Dietary Interview Technical Support File - Food Codes", "DRXFCD_H", "2.2 MB", "September 2016",
    "Dietary Supplement Use 24-Hour - Individual Dietary Supplements, First Day", "DS1IDS_H", "3.2 MB", "December 2016",
    "Dietary Supplement Use 24-Hour - Individual Dietary Supplements, Second Day", "DS2IDS_H", "3.2 MB", "December 2016",
    "Dietary Supplement Use 24-Hour - Total Dietary Supplements, First Day", "DS1TOT_H", "3.7 MB", "December 2016",
    "Dietary Supplement Use 24-Hour - Total Dietary Supplements, Second Day", "DS2TOT_H", "3.7 MB", "December 2016",
    "Dietary Supplement Use 30 Day - Individual Dietary Supplements", "DSQIDS_H", "7.7 MB", "December 2016",
    "Dietary Supplement Use 30-Day - Total Dietary Supplements", "DSQTOT_H", "3 MB", "December 2016"
  )
  ret <- setDataFileInfo(x, path)
  
  x <- c(
    "Dietary Supplement Database - Blend Information", "DSBI", "8.8 MB", "January 2022", 
    "Dietary Supplement Database - Ingredient Information", "DSII", "74 MB", "January 2022",
    "Dietary Supplement Database - Product Information", "DSPI", "5.7 MB", "January 2022"
  )
  ret2 <- setDataFileInfo(x, path2)
  
  rbind(ret, ret2)
  
}

dataFileInfo_2013_exam <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2013-2014/"
  
  x <- c(
    "Blood Pressure",	"BPX_H", "1.7 MB", "October 2015",
    "Body Measures",      "BMX_H", "2 MB", "October 2015",
    "Dual-Energy X-ray Absorptiometry - Abdominal Aortic Calcification", "DXXAAC_H", "582.8 KB", "September 2018",
    "Dual-Energy X-ray Absorptiometry - Android/Gynoid Measurements",    "DXXAG_H",  "1.1 MB",   "October 2021", 
    "Dual-Energy X-ray Absorptiometry - Femur",                          "DXXFEM_H", "582.8 KB", "October 2015",
    "Dual-Energy X-ray Absorptiometry - FRAX Score",                     "DXXFRX_H", "175.4 KB", "January 2016",
    "Dual-Energy X-ray Absorptiometry - L1 Vertebrae Morphology",	       "DXXL1_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - L2 Vertebrae Morphology",        "DXXL2_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - L3 Vertebrae Morphology",        "DXXL3_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - L4 Vertebrae Morphology",        "DXXL4_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - Spine",                          "DXXSPN_H", "699.3 KB", "October 2015",
    "Dual-Energy X-ray Absorptiometry - T10 Vertebrae Morphology",       "DXXT10_H", "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T11 Vertebrae Morphology",       "DXXT11_H", "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T12 Vertebrae Morphology",       "DXXT12_H", "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T4 Vertebrae Morphology",        "DXXT4_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T5 Vertebrae Morphology",        "DXXT5_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T6 Vertebrae Morphology",        "DXXT6_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T7 Vertebrae Morphology",        "DXXT7_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T8 Vertebrae Morphology",        "DXXT8_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - T9 Vertebrae Morphology",        "DXXT9_H",  "6.2 MB",   "December 2018",
    "Dual-Energy X-ray Absorptiometry - Vertebral Fracture Assessment",  "DXXVFA_H", "844.8 KB", "October 2015",
    "Dual-Energy X-ray Absorptiometry - Whole Body",                     "DXX_H",    "4.3 MB",   "March 2020",
    "Fluorosis - Clinical",                                              "FLXCLN_H", "617.1 KB", "April 2019",
    "Muscle Strength - Grip Test",                                       "MGX_H",    "1.8 MB",   "June 2016",
    "Oral Health - Dentition",                                           "OHXDEN_H", "7 MB",     "August 2016",
    "Oral Health - Periodontal",                                         "OHXPER_H", "18.2 MB",  "August 2016",
    "Oral Health - Recommendation of Care",                              "OHXREF_H", "885.7 KB", "August 2016",
    "Physical Activity Monitor - Day",                                   "PAXDAY_H", "7 MB",     "November 2020",
    "Physical Activity Monitor - Header",                                "PAXHD_H",  "428.5 KB", "November 2020",
    "Physical Activity Monitor - Hour",                                  "PAXHR_H",  "138 MB",   "November 2020",
    "Physical Activity Monitor - Minute",                                "PAXMIN_H", "8.7 GB",   "November 2020",
    #"Physical Activity Monitor - Raw data",                              "FTP",      "1.17 TB",  "January 2022",
    "Taste & Smell",                                                     "CSX_H",    "1.1 MB",   "February 2016")
  
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2013_lab <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2013-2014/"
  
  x <- c(
    "Acrylamide & Glycidamide", "AMDGYD_H", "130.7 KB", "December 2019",
    "Acrylamide & Glycidamide - Special Sample", "AMDGDS_H", "123.7 KB", "December 2019", 
    "Albumin & Creatinine - Urine", "ALB_CR_H", "390.2 KB", "September 2016",
    "Aldehydes - Serum", "ALD_H", "476.4 KB", "October 2019",
    "Aldehydes - Serum - Special Sample", "ALDS_H", "533.4 KB", "October 2019",
    "Apolipoprotein B", "APOB_H", "105.3 KB", "February 2016",
    "Aromatic Amines - Urine", "AA_H", "304 KB", "February 2020",
    "Aromatic Amines - Urine - Special Sample", "AAS_H", "287.6 KB", "February 2020", 
    "Arsenic - Total - Urine", "UTAS_H", "109.1 KB", "February 2022",
    "Arsenic - Total - Urine - Special Sample", "UTASS_H", "103.2 KB", "February 2022",
    "Arsenics - Speciated - Urine", "UAS_H", "325.7 KB", "January 2016",
    "Arsenics - Speciated - Urine - Special Sample", "UASS_H", "308.1 KB", "December 2015",
    "Brominated Flame Retardants (BFRs) - Pooled Samples", "BFRPOL_H", "99.7 KB", "November 2020",
    "Caffeine & Caffeine Metabolites - Urine", "CAFE_H", "694.8 KB", "June 2020",
    "Chlamydia - Urine", "CHLMDA_H", "35.7 KB", "October 2015",
    "Chlamydia Pgp3 (plasmid gene product 3) ELISA (enzyme linked immunosorbent assay) and multiplex bead array (MBA) results", "SSCT_H", "37.3 KB", "February 2021",
    "Cholesterol - HDL", "HDL_H", "195.5 KB", "March 2016",
    "Cholesterol - LDL & Triglycerides", "TRIGLY_H", "157.7 KB", "March 2016",
    "Cholesterol - Total", "TCHOL_H", "195.5 KB", "March 2016",
    "Complete Blood Count with 5-part Differential - Whole Blood", "CBC_H", "1.5 MB", "July 2016",
    "Copper, Selenium & Zinc - Serum", "CUSEZN_H", "195.7 KB", "January 2016",
    "Cotinine and Hydroxycotinine - Serum", "COT_H", "349.6 KB", "October 2017",
    "Cotinine, Hydroxycotinine, & Other Nicotine Metabolites and Analogs - Urine", "UCOT_H", "455.6 KB", "May 2019",
    "Cotinine, Hydroxycotinine, & Other Nicotine Metabolites and Analogs - Urine - Special Sample", "UCOTS_H", "431 KB", "May 2019", 
    "DEET and Metabolites", "DEET_H", "196 KB", "March 2017",
    "Fasting Questionnaire", "FASTQX_H", "1.4 MB", "October 2015",
    "Fluoride - Plasma", "FLDEP_H", "64.5 KB", "November 2019", 
    "Fluoride - Water", "FLDEW_H", "100.2 KB", "November 2019", 
    "Folate - RBC", "FOLATE_H", "222 KB", "February 2019",
    "Folate Forms - Total & Individual - Serum", "FOLFMS_H", "1.1 MB", "February 2019",
    "Formaldehyde", "FORMAL_H", "87.3 KB", "March 2020", 
    "Formaldehyde - Special Sample", "FORMAS_H", "82.7 KB", "March 2020", 
    "Glycohemoglobin", "GHB_H", "110.1 KB", "October 2015",
    "Hepatitis A", "HEPA_H", "144.4 KB", "October 2015",
    "Hepatitis B: core antibody, surface antigen, and Hepatitis D Antibody", "HEPBD_H", "260.4 KB", "October 2015",
    "Hepatitis B Surface Antibody", "HEPB_S_H", "144.4 KB", "October 2015",
    "Hepatitis C: Confirmed Antibody (INNO-LIA)", "SSHEPC_H", "112.8 KB", "November 2021",
    "Hepatitis C: RNA (HCV-RNA) and Hepatitis C Genotype", "HEPC_H", "195.5 KB", "September 2017", 
    "Hepatitis E : IgG & IgM Antibodies", "HEPE_H", "195.5 KB", "October 2015",
    "Herpes Simplex Virus Type-1 & Type-2", "HSV_H", "93.4 KB", "October 2015",
    "Heterocyclic Aromatic Amines - Urine", "HCAA_H", "477.3 KB", "September 2017", 
    "Heterocyclic Aromatic Amines (HCAA) - Urine - Special Sample", "HCAAS_H", "451.5 KB", "September 2017", 
    "HIV Antibody Test", "HIV_H", "65.7 KB", "October 2015",
    "Human Papillomavirus (HPV) - Oral Rinse", "ORHPV_H", "1.6 MB", "October 2016",
    "Human Papillomavirus (HPV) DNA - Vaginal Swab: Roche Cobas & Roche Linear Array", "HPVSWR_H", "716.6 KB", "December 2016",
    "Human Papillomavirus (HPV) DNA Results from Penile Swab Samples: Roche Linear Array", "HPVP_H", "639 KB", "June 2016",
    "Insulin", "INS_H", "157.7 KB", "April 2016", 
    "Iodine - Urine", "UIO_H", "87.3 KB", "January 2016",
    "Klotho - Serum (Surplus)", "SSKL_H", "44.3 KB", "April 2021",
    "Lead, Cadmium, Total Mercury, Selenium, and Manganese - Blood", "PBCD_H", "790.9 KB", "January 2016",
    "Mercury: Inorganic - Urine", "UHG_H", "109.1 KB", "January 2016",
    "Mercury: Inorganic, Ethyl and Methyl - Blood", "IHGEM_H", "419.1 KB", "January 2016",
    "Metals - Urine", "UM_H", "628.9 KB", "January 2016",
    "Metals - Urine - Special Sample", "UMS_H", "594.9 KB", "December 2015",
    "Methylmalonic Acid", "MMA_H", "135.6 KB", "October 2018",
    "Non-dioxin-like Polychlorinated Biphenyls & Mono-ortho-substituted Polychlorinated Biphenyls - Serum - Pooled Samples", "PCBPOL_H", "184.5 KB", "December 2019",
    "Oral Glucose Tolerance Test", "OGTT_H", "275.1 KB", "January 2016",
    "Perchlorate, Nitrate & Thiocyanate - Urine", "PERNT_H", "195.7 KB", "June 2016",
    "Perchlorate, Nitrate & Thiocyanate - Urine - Special Sample", "PERNTS_H", "185.2 KB", "June 2016",
    "Perfluoroalkyl and Polyfluoroalkyl Substances (formerly Polyfluoroalkyl Chemicals - PFC)", "PFAS_H", "332.2 KB", "July 2016",
    "Personal Care and Consumer Product Chemicals and Metabolites", "EPHPP_H", "590.2 KB", "September 2016", 
    "Pesticides - Organochlorine Pesticides - Serum - Pooled Samples", "PSTPOL_H", "64.4 KB", "December 2019",
    "Phthalates & Plasticizers Metabolites - Urine", "PHTHTE_H", "633.9 KB", "December 2016",
    "Plasma Fasting Glucose", "GLU_H", "157.7 KB", "January 2016",
    "Polyaromatic Hydrocarbons (PAH) - Urine", "PAH_H", "369 KB", "December 2016",
    "Pooled-Sample Technical Support File", "POOLTF_H", "55.5 KB", "May 2018",
    "Pregnancy Test - Urine", "UCPREG_H", "20.8 KB", "October 2015",
    "Pyrethroids, Herbicides, & Organophosphorus Metabolites - Urine", "UPHOPM_H", "347.8 KB", "May 2020",
    "Sex Steroid Hormone - Serum", "TST_H", "455.2 KB", "November 2018", 
    "Standard Biochemistry Profile", "BIOPRO_H", "2 MB", "October 2015",
    "Tissue Transglutaminase Assay (IgA-TTG) & IgA Endomyseal Antibody Assay (IgA EMA)", "TGEMA_H", "195.5 KB", "October 2015",
    "Tobacco-specific Nitrosamines (TSNAs) - Urine", "TSNA_H", "584.9 KB", "April 2019",
    "Trichomonas - Urine", "TRICH_H", "65.7 KB", "December 2016",
    "Urine Flow Rate", "UCFLOW_H", "649.8 KB", "April 2018",
    "Vitamin B12", "VITB12_H", "135.6 KB", "August 2017",
    "Vitamin D", "VID_H", "664.5 KB", "May 2018",
    "Volatile N-Nitrosamine Compounds (VNAs) - Urine", "VNA_H", "260.6 KB", "December 2018",
    "Volatile N-Nitrosamine Compounds (VNAs) - Urine - Special Sample", "VNAS_H", "246.6 KB", "December 2018", 
    "Volatile Organic Compound (VOC) Metabolites - Urine", "UVOC_H", "1.2 MB", "November 2021",
    "Volatile Organic Compound (VOC) Metabolites - Urine - Special Sample", "UVOCS_H", "1.1 MB", "November 2021",
    "Volatile Organic Compounds (VOCs) and Trihalomethanes/MTBE - Blood - Special Sample", "VOCWBS_H", "2.6 MB", "December 2017", 
    "Volatile Organic Compounds and Trihalomethanes/MTBE - Blood", "VOCWB_H", "2.1 MB", "February 2017"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_2013_question <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nhanes/2013-2014/"
  
  x <- c(
    "Acculturation", "ACQ_H", "434.2 KB", "MArch 2016",
    "Alcohol Use", "ALQ_H", "464.9 KB", "March 2016",
    "Blood Pressure & Cholesterol", "BPQ_H", "709.7 KB", "October 2015",
    "Cardiovascular Health", "CDQ_H", "509.8 KB", "October 2015",
    "Cognitive Functioning", "CFQ_H", "268.4 KB", "March 2017",
    "Consumer Behavior", "CBQ_H", "478.5 KB", "November 2015",
    "Creatine Kinase", "CKQ_H", "438 KB", "February 2017",
    "Current Health Status", "HSQ_H", "664.5 KB", "December 2015",
    "Dermatology", "DEQ_H", "278.2 KB", "August 2017",
    "Diabetes", "DIQ_H", "4 MB", "October 2015",
    "Diet Behavior & Nutrition", "DBQ_H", "4 MB", "November 2015",
    "Disability", "DLQ_H", "536 KB", "October 2015",
    "Drug Use", "DUQ_H", "1.6 MB", "March 2016",
    "Early Childhood", "ECQ_H", "292 KB", "October 2015",
    "Food Security", "FSQ_H", "3.1 MB", "May 2016",
    "Health Insurance", "HIQ_H", "1.3 MB", "October 2015",
    "Hepatitis", "HEQ_H", "336.2 KB", "October 2015",
    "Hospital Utilization & Access to Care", "HUQ_H", "797 KB", "October 2015",
    "Housing Characteristics", "HOQ_H", "239.7 KB", "October 2015",
    "Immunization", "IMQ_H", "637.7 KB", "October 2015",
    "Income", "INQ_H", "1.2 MB", "October 2015",
    "Kidney Conditions - Urology", "KIQ_U_H", "724.1 KB", "February 2017",
    "Medical Conditions", "MCQ_H", "7.1 MB", "October 2015",
    "Mental Health - Depression Screener", "DPQ_H", "511.4 KB", "March 2016",
    "Occupation", "OCQ_H", "659.1 KB", "December 2019",
    "Oral Health", "OHQ_H", "4 MB", "August 2016",
    "Osteoporosis", "OSQ_H", "2.7 MB", "October 2015",
    "Pesticide Use", "PUQMEC_H", "195.5 KB", "October 2015",
    "Physical Activity", "PAQ_H", "7 MB", "March 2017",
    "Physical Functioning", "PFQ_H", "2.5 MB", "November 2016",
    "Prescription Medications", "RXQ_RX_H", "9.4 MB", "December 2016",
    "Prescription Medications - Drug Information", "RXQ_DRUG", "2.6 KB", "September 2021",
    "Preventive Aspirin Use", "RXQASA_H", "240.2 KB", "December 2016",
    "Reproductive Health", "RHQ_H", "1.3 MB", "December 2017",
    "Sexual Behavior", "SXQ_H", "1.9 MB", "October 2015",
    "Sleep Disorders", "SLQ_H", "203.3 KB", "February 2016",
    "Smoking - Cigarette Use", "SMQ_H", "2.1 MB", "September 2016",
    "Smoking - Household Smokers", "SMQFAM_H", "319.2 KB", "October 2015",
    "Smoking - Recent Tobacco Use", "SMQRTU_H", "1.5 MB", "February 2018",
    "Smoking - Secondhand Smoke Exposure", "SMQSHS_H", "1.1 MB", "October 2015", 
    "Taste & Smell", "CSQ_H", "1.1 MB", "March 2016",
    "Volatile Toxicant (Subsample)", "VTQ_H", "658.2 KB", "February 2017",
    "Weight History", "WHQ_H", "1.7 MB", "March 2016",
    "Weight History - Youth", "WHQMEC_H", "49.8 KB", "March 2016"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}


dataFileInfo_NNYFS_2012 <- function(type) {
  
  base <- NULL
  nms  <- names(type)
 
  if ("DE" %in% type) {
    new  <- dataFileInfo_NNYFS_demo() 
    base <- updateDataFileInfo(base, new, nms[type=="DE"])
  }
  if ("DI" %in% type) {
    new  <- dataFileInfo_NNYFS_diet() 
    base <- updateDataFileInfo(base, new, nms[type=="DI"])
  }
  if ("EX" %in% type) {
    new  <- dataFileInfo_NNYFS_exam() 
    base <- updateDataFileInfo(base, new, nms[type=="EX"])
  }
  if ("QU" %in% type) {
    new  <- dataFileInfo_NNYFS_question() 
    base <- updateDataFileInfo(base, new, nms[type=="QU"])
  }
  
  base
  
}

dataFileInfo_NNYFS_exam <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nnyfs/"
  
  x <- c(
    "Body Measures",                        "Y_BMX",    "274.7 KB", "September 2013",
    "Cardiorespiratory Endurance",          "Y_CEX",    "393.1 KB", "January 2016",
    "Cardiovascular Fitness",               "Y_CVX",    "132.1 KB", "April 2014",
    "Lower Body Muscle Strength",           "Y_LMX",    "1.1 MB",   "September 2013",
    "Modified Pull-Up",                     "Y_MPX",    "43.3 KB",  "September 2013",
    "Muscle Strength - Grip Test",          "Y_MGX",    "204.5 KB", "September 2013",
    "Physical Activity Monitor - Day",      "Y_PAXDAY", "1.3 MB",   "November 2020",
    "Physical Activity Monitor - Header",   "Y_PAXHD",  "77.4 KB",  "November 2020",
    "Physical Activity Monitor - Hour",     "Y_PAXHR",  "26.3 MB",  "November 2020",
    "Physical Activity Monitor - Minute",   "Y_PAXMIN", "1.6 GB",   "November 2020",
    #"Physical Activity Monitor - Raw data", "FTP",      "245 GB",	"January 2022",
    "Plank",                                "Y_PLX",    "50.5 KB",  "September 2013",
    "Test of Gross Motor Development",      "Y_GMX",    "165.3 KB", "January 2016")
  
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_NNYFS_demo <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nnyfs/"
  
  x <- c(
    "Demographic Variables & Sample Weights", "Y_DEMO",  "350.4 KB", "January 2015"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_NNYFS_diet <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nnyfs/"
  
  x <- c(
    "Dietary Interview - Individual Foods", "Y_DR1IFF", "14.3 MB", "September 2014",
    "Dietary Interview - Total Nutrient Intakes", "Y_DR1TOT", "1.2 MB", "September 2014",
    "Dietary Supplement Database - Blend Information", "DSBI", "4.4 MB", "September 2014",
    "Dietary Supplement Database - Ingredient Information", "DSII", "31 MB", "September 2014",
    "Dietary Supplement Database - Product Information", "DSPI", "2.9 MB", "September 2014",
    "Dietary Supplement Use 24-Hour - Individual Dietary Supplements", "Y_DS1IDS", "224.9 KB", "September 2014", 
    "Dietary Supplement Use 24-Hour - Total Dietary Supplements", "Y_DS1TOT", "573.4 KB", "September 2014",
    "Dietary Supplement Use 30 Day - Individual Dietary Supplements", "Y_DSQIDS", "645.7 KB", "January 2015",
    "Dietary Supplement Use 30-Day - Total Dietary Supplements", "Y_DSQTOT", "505.8 KB", "January 2015"
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}

dataFileInfo_NNYFS_question <- function() {
  
  path <- "https://wwwn.cdc.gov/Nchs/Nnyfs/"
  
  x <- c(
    "Acculturation", "Y_ACQ", "65.5 KB", "August 2014",
    "Diabetes", "Y_DIQ", "78.4 KB", "September 2013",
    "Diet Behavior & Nutrition", "Y_DBQ", "156.1 KB", "February 2015",
    "Early Childhood", "Y_ECQ", "91.4 KB", "September 2013",
    "Health Insurance", "Y_HIQ", "169.1 KB", "September 2013", 
    "Hospital Utilization & Access to Care", "Y_HUQ", "52.5 KB", "September 2013",
    "Medical Conditions", "Y_MCQ", "91.4 KB", "September 2013",
    "Physical Activity", "Y_PAQ", "1.2 MB", "September 2013", 
    "Physical Functioning", "Y_PFQ", "65.5 KB", "September 2013",
    "Prescription Medications", "Y_RXQ_RX", "288.4 KB", "July 2014",
    "Respiratory Health", "Y_RDQ", "39.6 KB", "September 2013",
    "Smoking - Cigarette Use", "Y_SMQ", "16.6 KB", "September 2013"  
  )
  ret <- setDataFileInfo(x, path)
  
  ret
  
}
