title1 "NHANES 2001-2004: Monte Carlo Distribution of Percent of Usual Energy Intake from Saturated Fat";

%let foodvar1  = DRXTSFAT;    /* name of the first food/nutrient variable (numerator of ratio) */
%let foodvar2  = DRXTKCAL;    /* name of the second food/nutrient variable (denominator of ratio) */
%let modeltype = ONEPART;     /* "onepart" specifies that the first food/nutrient is consumed every day. */
                              /* note: the second food/nutrient is always assumed to be consumed every day. */

%let seed = 123457;           /* seed for random number generator for Monte Carlo simulation */
%global seed;                 /* allow macro variable seed to be available to macro distrib_bivariate. */

libname outlib "c:\projects\usual.intake\nhanes\outlib";

%include "c:\projects\usual.intake\macros\distrib_bivariate.macro.v1.1.sas";
%include "c:\projects\usual.intake\macros\percentiles_survey.macro.v1.1.sas";

proc format;
  value sexfmt
    1 = "Male"
    2 = "Female";
  value stratfmt
    1 = "Children 1-8"
    2 = "Males 9+"
    3 = "Females 9+";
  value subfmt
    1 = "Children 1-3"
    2 = "Children 4-8"
    3 = "Males 9-13"
    4 = "Males 14-18"
    5 = "Males 19-30"
    6 = "Males 31-50"
    7 = "Males 51-70"
    8 = "Males 71+"
    9 = "Males 19+"
   10 = "Females 9-13"
   11 = "Females 14-18"
   12 = "Females 19-30"
   13 = "Females 31-50"
   14 = "Females 51-70"
   15 = "Females 71+"
   16 = "Females 19+"
   17 = "All 1+";
  run;

 /* get parameter estimates and predicted values calculated by macro NLMixed_Bivariate. */

data parms;
  set outlib.parms_b_&foodvar1._&foodvar2;
  format stratum stratfmt.;
  run;

data pred;
  set outlib.pred_b_&foodvar1._&foodvar2;
  format sex sexfmt. stratum stratfmt.;
  run;

proc sort data=pred;  by stratum id repeat;  run;

 /* for predicted values, create two records per subject, one for weekday and one for weekend. */

data pred;
  merge pred parms(keep=stratum a1_repeat2 a2_repeat2 a1_weekend a2_weekend);
    by stratum;
  run;

data pred (drop=a1_repeat2 a2_repeat2 a1_weekend a2_weekend);
  set pred;
    by stratum id;

  if (first.id);  /* keep first record per subject. */

 /* the following code assumes that "repeat2" is a covariate in the model:            */
 /*   repeat2 = 0 for the first administered 24-hour dietary recall (drddaycd = 1).   */
 /*   repeat2 = 1 for the second administered 24-hour dietary recall (drddaycd = 2).  */

 /* for each subject, calculate predicted value when repeat2 = 0. */

  if (repeat2 = 1) then do;
    repeat  = 1;
    repeat2 = 0;
    pred_x_a1 = pred_x_a1 - a1_repeat2;
    pred_x_a2 = pred_x_a2 - a2_repeat2;
    end;

 /* the following code assumes that "weekend" is a covariate in the model:  */
 /*   weekend = 0 if the 24-hour dietary recall was for Monday-Thursday.    */
 /*   weekend = 1 if the 24-hour dietary recall was for Friday-Sunday.      */

 /* create record for weekend = 0. */

  if (weekend = 1) then do;
    weekend = 0;
    pred_x_a1 = pred_x_a1 - a1_weekend;
    pred_x_a2 = pred_x_a2 - a2_weekend;
    end;

  day_wgt = 4/7;
  output;

 /* create record for weekend = 1. */

  weekend = 1;
  pred_x_a1 = pred_x_a1 + a1_weekend;
  pred_x_a2 = pred_x_a2 + a2_weekend;
  day_wgt = 3/7;
  output;
  run;

 /* for each food/nutrient, create macro variable that equals half the minimum nonzero amount. */

data _null_;
  set pred;
    by stratum;
  if (first.stratum);

  min_a1 = min_a1 / 2;
  if (stratum = 1) then call symput("min_a11",trim(left(put(min_a1, best12.))));
  if (stratum = 2) then call symput("min_a21",trim(left(put(min_a1, best12.))));
  if (stratum = 3) then call symput("min_a31",trim(left(put(min_a1, best12.))));

  min_a2 = min_a2 / 2;
  if (stratum = 1) then call symput("min_a12",trim(left(put(min_a2, best12.))));
  if (stratum = 2) then call symput("min_a22",trim(left(put(min_a2, best12.))));
  if (stratum = 3) then call symput("min_a32",trim(left(put(min_a2, best12.))));
 run;

 /* create separate data set for each stratum. */

data parms1 parms2 parms3;
  set parms;
  if (stratum = 1) then output parms1;
  if (stratum = 2) then output parms2;
  if (stratum = 3) then output parms3;
  run;

data pred1 pred2 pred3;
  set pred;
  if (stratum = 1) then output pred1;
  if (stratum = 2) then output pred2;
  if (stratum = 3) then output pred3;
  run;

 /* delete unneeded data sets. */

proc datasets lib=work nolist;
  delete parms pred;
  run;
  quit;

 /* call macro Distrib_Bivariate to generate monte carlo distribution of usual intake. */

title2 "Stratum 1 = Chilren 1-8";

%Distrib_Bivariate (param          = parms1,
                    predicted      = pred1,
                    subject        = id,
                    modeltype      = &modeltype,
                    nsim_mc        = 100,
                    day_wgt        = day_wgt,
                    min_a1         = &min_a11,
                    min_a2         = &min_a12,
                    print          = N,
                    ntitle         = 2);

options notes;

data mcsim;
  set _mcsim;
  run;

title2 "Stratum 2 = Males 9+";

%Distrib_Bivariate (param          = parms2,
                    predicted      = pred2,
                    subject        = id,
                    modeltype      = &modeltype,
                    nsim_mc        = 100,
                    day_wgt        = day_wgt,
                    min_a1         = &min_a21,
                    min_a2         = &min_a22,
                    print          = N,
                    ntitle         = 2);

options notes;

data mcsim;
  set mcsim _mcsim;
    by stratum;
  run;

title2 "Stratum 3 = Females 9+";

%Distrib_Bivariate (param          = parms3,
                    predicted      = pred3,
                    subject        = id,
                    modeltype      = &modeltype,
                    nsim_mc        = 100,
                    day_wgt        = day_wgt,
                    min_a1         = &min_a31,
                    min_a2         = &min_a32,
                    print          = N,
                    ntitle         = 2);

options notes;

data mcsim;
  set mcsim _mcsim;
    by stratum;
  run;

proc datasets lib=work nolist;
  delete _mcsim;
  run;
  quit;

title2;

 /* calculate usual nutrient density. */

data mcsim;
  set mcsim;

 /* define nutrient density. */

  *t_density = 1000 * (t1 / t2);          /* nutrient/food per 1000 kcal         */
  t_density =  100 * ((9 * t1) / t2);    /* percent calories from saturated fat */

  run;

 /* call macro Percentiles_Survey to calculate percentiles of usual nutrient density. */

 /* table 1: by age and sex */

title2 "Table 1: Percentiles by Age and Sex";

data mcsim2;
  set mcsim;

  Subpopulation = agegrp;
  if (sex = 2 & agegrp >= 3) then Subpopulation = Subpopulation + 7;
  format Subpopulation subfmt.;
  run;

%Percentiles_Survey (data       = mcsim2,
                     byvar      = Subpopulation,
                     var        = t_density,
                     weight     = rndw,
                     cutpoints  = 10 12 15,
                     print      = N,
                     ntitle     = 2);

data pctl;
  set _percentiles;
    by Subpopulation;
  run;

 /* table 2: adults 19+, by gender */

title2 "Table 2: Percentiles for Adults 19+, by Sex";

data mcsim2;
  set mcsim;

  if (age < 19) then delete;
       if (sex = 1) then Subpopulation = 9;
  else if (sex = 2) then Subpopulation = 16;
  format Subpopulation subfmt.;
  run;

%Percentiles_Survey (data       = mcsim2,
                     byvar      = Subpopulation,
                     var        = t_density,
                     weight     = rndw,
                     cutpoints  = 10 12 15,
                     print      = N,
                     ntitle     = 2);

data pctl;
  set pctl _percentiles;
    by subpopulation;
  run;

 /* table 3: all persons 1+ */

title2 "Table 3: Percentiles for All Persons 1+";

data mcsim2;
  set mcsim;

  Subpopulation = 17;
  format Subpopulation subfmt.;
  run;

%Percentiles_Survey (data       = mcsim2,
                     byvar      = Subpopulation,
                     var        = t_density,
                     weight     = rndw,
                     cutpoints  = 10 12 15,
                     print      = N,
                     ntitle     = 2);

data pctl;
  set pctl _percentiles;
    by subpopulation;
  run;

 /* print summary tables by subpopulation. */

title2 "Estimated Mean, Standard Deviation and Percentiles";
title3 "By Subpopulation";

proc print data=pctl label;
  id subpopulation;
  var Mean StdDev Pctile5 Pctile10 Pctile25 Pctile50 Pctile75 Pctile90 Pctile95;
  format Mean StdDev Pctile5 Pctile10 Pctile25 Pctile50 Pctile75 Pctile90 Pctile95 7.2;
  run;

title2 "Estimated Cut-Point Probabilities";
title3 "By Subpopulation";

proc print data=pctl label;
  id subpopulation;
  var Prob1-Prob3;
  format Prob1-Prob3 7.2;
  label Prob1 = "Prob(X <= 10)"
        Prob2 = "Prob(X <= 12)"
        Prob3 = "Prob(X <= 15)";
  run;

 /* save means, percentiles and cut-point probabilities. */

data outlib.pctl_b_&foodvar1._&foodvar2;
  set pctl;
  run;
