title1 "NHANES 2001-2004: Univariate Nonlinear Mixed Model of Total Energy (kcal/day) Intake";

%let foodvar   = DRXTKCAL;    /* name of the food/nutrient variable */
%let modeltype = ONEPART;     /* "onepart" specifies that the food/nutrient is consumed every day. */

%let nloptions = technique=trureg maxfunc=10000 maxiter=200;    /* options for the SAS NLMIXED procedure */

libname lib    "c:\projects\usual.intake\nhanes";
libname outlib "c:\projects\usual.intake\nhanes\outlib";

%include "c:\projects\usual.intake\macros\nlmixed_univariate.macro.v1.1.sas";
%include "c:\projects\usual.intake\macros\boxcox_survey.macro.v1.1.sas";

proc format;
  value sexfmt
    1 = "Male"
    2 = "Female";
  value stratfmt
    1 = "Children 1-8"
    2 = "Males 9+"
    3 = "Females 9+";
  run;

 /* get nhanes data. */

data nhanes (keep=stratum id sex age race repeat dayofweek weekend agegrp
                  rndw /*rndw1-rndw32*/ recall_food);

  set lib.t_0104nc_may7;

  stratum   = stra;
  id        = seqn;
  sex       = riagendr;
  age       = ridageyr;
  race      = ridreth1;
  repeat    = drddaycd;
  dayofweek = drdday;
  weekend   = wkend;

  if (stratum < 1 or stratum > 3) then delete;

  recall_food = &foodvar;

  format sex sexfmt. stratum stratfmt.;
  run;

proc sort data=nhanes;  by stratum id repeat;  run;

 /* create covariates for model. */

data nhanes;
  set nhanes;

  if (repeat = 2) then repeat2 = 1;
  else repeat2 = 0;

  if (sex = 2) then female = 1;
  else if (sex = 1) then female = 0;

  array a (*) age_1to3 age_4to8 age_9to13 age_14to18 age_19to30 age_31to50 age_51to70 age_71plus;
  do i = 1 to dim(a);  a(i) = 0;  end;
  a(agegrp) = 1;
  run;

 /* get minimum amount on consumption day. */

proc means data=nhanes noprint;
  by stratum;
  where (recall_food > 0);
  var recall_food;
  output out=min_a(keep=stratum min_a) min=min_a;
  run;

 /* add minimum amount on consumption day to data set.  */
 /* for one-part models, set 24hr zero values to half the minimum nonzero value. */

data nhanes;
  merge nhanes min_a;
    by stratum;
  modeltype = upcase("&modeltype");
  if (modeltype = "ONEPART" & recall_food = 0) then recall_food = min_a / 2;
  run;

 /* find best box-cox transformation for 24hr. */

data nhanes_boxcox;
  set nhanes;
    by stratum id;
  if (first.id);    /* keep first record per subject. */
  if (modeltype ^= "ONEPART" & recall_food = 0) then delete;  /* for two-part models, delete zero values. */
  run;

 /* call macro boxcox_survey to find best box-cox transformation to normality. */

%BoxCox_Survey (data           = nhanes_boxcox,
                byvar          = stratum,
                subject        = id,
                var            = recall_food,
                weight         = rndw,
                print          = Y,
                ntitle         = 2);

 /* create macro variables for the box-cox transformations. */

data _null;
  set _lambda;
  if (stratum = 1) then call symput("lambda1",trim(left(put(lambda_recall_food,4.2))));
  if (stratum = 2) then call symput("lambda2",trim(left(put(lambda_recall_food,4.2))));
  if (stratum = 3) then call symput("lambda3",trim(left(put(lambda_recall_food,4.2))));
  run;

 /* create separate data sets each stratum (children 1-8, males 9+, females 9+). */

data nhanes1 nhanes2 nhanes3;
  set nhanes;
  if (stratum = 1) then output nhanes1;
  if (stratum = 2) then output nhanes2;
  if (stratum = 3) then output nhanes3;
  run;

 /* delete unneeded data sets. */

proc datasets lib=work nolist;
  delete nhanes;
  run;
  quit;

 /* call macro NLMIxed_Univariate to fit nonlinear mixed model in each stratum. */

 /* in this example, we specify a fixed value for parameter lambda (box-cox   */
 /* transformation of amount consumed). Alternatively, one can let the macro  */
 /* estimate lambda by not specifying a value for macro parameter "lambda".   */

title2 "Stratum 1 = Chilren 1-8";

%NLMIxed_Univariate (data           = nhanes1,
                     subject        = id,
                     repeat         = repeat,
                     response       = recall_food,
                     modeltype      = &modeltype,
                     covars_prob    = repeat2 weekend female age_4to8,
                     covars_amt     = repeat2 weekend female age_4to8,
                     lambda         = &lambda1,
                     replicate_var  = rndw,
                     nloptions      = &nloptions,
                     print          = Y,
                     ntitle         = 2);

data parms_u1;
  stratum = 1;
  set parms_u;
  run;

title2 "Stratum 2 = Males 9+";

%NLMIxed_Univariate (data           = nhanes2,
                     subject        = id,
                     repeat         = repeat,
                     response       = recall_food,
                     modeltype      = &modeltype,
                     covars_prob    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                     covars_amt     = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                     lambda         = &lambda2,
                     replicate_var  = rndw,
                     nloptions      = &nloptions,
                     print          = Y,
                     ntitle         = 2);

data parms_u2;
  stratum = 2;
  set parms_u;
  run;

title2 "Stratum 3 = Females 9+";

%NLMIxed_Univariate (data           = nhanes3,
                     subject        = id,
                     repeat         = repeat,
                     response       = recall_food,
                     modeltype      = &modeltype,
                     covars_prob    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                     covars_amt     = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                     lambda         = &lambda3,
                     replicate_var  = rndw,
                     nloptions      = &nloptions,
                     print          = Y,
                     ntitle         = 2);

data parms_u3;
  stratum = 3;
  set parms_u;
  run;

 /* combine data sets of parameter estimates. */

data parms_u;
  set parms_u1 parms_u2 parms_u3;
    by stratum;
  run;

 /* keep minimum amount on consumption day. */

data parms_u;
  merge parms_u min_a;
    by stratum;
  run;

 /* save parameter estimates. */

data outlib.parms_u_&foodvar;
  set parms_u;
  run;
