title1 "NHANES 2001-2004: Nonlinear Mixed Model of Saturated Fat (g/day) and Total Energy (kcal/day) Intake";

%let foodvar1  = DRXTSFAT;    /* name of the first food/nutrient variable (numerator of ratio) */
%let foodvar2  = DRXTKCAL;    /* name of the second food/nutrient variable (denominator of ratio) */
%let modeltype = ONEPART;     /* "onepart" specifies that the first food/nutrient is consumed every day. */
                              /* note: the second food/nutrient is always assumed to be consumed every day. */

%let nloptions = technique=trureg maxfunc=10000 maxiter=200;    /* options for the SAS NLMIXED procedure */

libname lib    "c:\projects\usual.intake\nhanes";
libname outlib "c:\projects\usual.intake\nhanes\outlib";

%include "c:\projects\usual.intake\macros\nlmixed_bivariate.macro.v1.1.sas";

proc format;
  value sexfmt
    1 = "Male"
    2 = "Female";
  value stratfmt
    1 = "Children 1-8"
    2 = "Males 9+"
    3 = "Females 9+";
  run;

 /* get nhanes data. */

data nhanes (keep=stratum id sex age race repeat dayofweek weekend agegrp
                  rndw /*rndw1-rndw32*/ recall_food1 recall_food2);

  set lib.t_0104nc_may7;

  stratum   = stra;
  id        = seqn;
  sex       = riagendr;
  age       = ridageyr;
  race      = ridreth1;
  repeat    = drddaycd;
  dayofweek = drdday;
  weekend   = wkend;

  if (stratum < 1 or stratum > 3) then delete;

  recall_food1 = &foodvar1;
  recall_food2 = &foodvar2;

  format sex sexfmt. stratum stratfmt.;
  run;

proc sort data=nhanes;  by stratum id repeat;  run;

 /* create covariates for model. */

data nhanes;
  set nhanes;

  if (repeat = 2) then repeat2 = 1;
  else repeat2 = 0;

  if (sex = 2) then female = 1;
  else if (sex = 1) then female = 0;

  array a (*) age_1to3 age_4to8 age_9to13 age_14to18 age_19to30 age_31to50 age_51to70 age_71plus;
  do i = 1 to dim(a);  a(i) = 0;  end;
  a(agegrp) = 1;
  run;

 /* get initial parameter estimates from univariate models. */

 /* initial parameter estimates for first food/nutrient. */

data init_parms_f1;
  set outlib.parms_u_&foodvar1;

 /* rename univariate parameters for bivariate model. */

  rename a_intercept        = A1_intercept
         a_repeat2          = A1_repeat2
         a_weekend          = A1_weekend
         a_female           = A1_female
         a_age_4to8         = A1_age_4to8
         a_age_14to18       = A1_age_14to18
         a_age_19to30       = A1_age_19to30
         a_age_31to50       = A1_age_31to50
         a_age_51to70       = A1_age_51to70
         a_age_71plus       = A1_age_71plus
         a_logsde           = A1_LogSDe
         a_lambda           = A1_Lambda
         logsdu2            = LogSDu2
         min_a              = min_a1;
  run;

 /* initial parameter estimates for second food/nutrient. */

data init_parms_f2;
  set outlib.parms_u_&foodvar2;

 /* rename univariate parameters for bivariate model. */

  rename a_intercept        = A2_intercept
         a_repeat2          = A2_repeat2
         a_weekend          = A2_weekend
         a_female           = A2_female
         a_age_4to8         = A2_age_4to8
         a_age_14to18       = A2_age_14to18
         a_age_19to30       = A2_age_19to30
         a_age_31to50       = A2_age_31to50
         a_age_51to70       = A2_age_51to70
         a_age_71plus       = A2_age_71plus
         a_logsde           = A2_LogSDe
         a_lambda           = A2_Lambda
         logsdu2            = LogSDu3
         min_a              = min_a2;
  run;

 /* combine initial parameter estimates for the two food/nutrients. */

data init_parms;
  merge init_parms_f1 init_parms_f2;
    by stratum;

  keep stratum
       A1_intercept A1_repeat2 A1_weekend A1_female A1_age_4to8 A1_age_14to18 A1_age_19to30 A1_age_31to50 A1_age_51to70 A1_age_71plus
       A2_intercept A2_repeat2 A2_weekend A2_female A2_age_4to8 A2_age_14to18 A2_age_19to30 A2_age_31to50 A2_age_51to70 A2_age_71plus
       A1_LogSDe A2_LogSDe
       LogSDu2 LogSDu3;

  format stratum stratfmt.;
  run;

 /* add minimum amount on consumption day to data set.  */
 /* for one-part models and for energy, set 24hr zero values to half the minimum nonzero value. */

data min_a;
  merge init_parms_f1 init_parms_f2;
    by stratum;

  keep stratum min_a1 min_a2;
  run;

data nhanes;
  merge nhanes min_a;
    by stratum;
  modeltype = upcase("&modeltype");
  if (modeltype = "ONEPART" & recall_food1 = 0) then recall_food1 = min_a1 / 2;
  if (recall_food2 = 0) then recall_food2 = min_a2 / 2;
  run;

 /* create macro variables for the box-cox transformations for the two food/nutrients. */

data lambdas;
  merge init_parms_f1 init_parms_f2;
    by stratum;

  if (stratum = 1) then call symput("lambda11",trim(left(put(a1_lambda,4.2))));
  if (stratum = 2) then call symput("lambda21",trim(left(put(a1_lambda,4.2))));
  if (stratum = 3) then call symput("lambda31",trim(left(put(a1_lambda,4.2))));

  if (stratum = 1) then call symput("lambda12",trim(left(put(a2_lambda,4.2))));
  if (stratum = 2) then call symput("lambda22",trim(left(put(a2_lambda,4.2))));
  if (stratum = 3) then call symput("lambda32",trim(left(put(a2_lambda,4.2))));

  keep stratum a1_lambda a2_lambda;
  run;

proc print data=lambdas;
  id stratum;
  var a1_lambda a2_lambda;
  title3 "Box-Cox Transformation Parameters";
  run;

title2;

 /* create separate data sets each stratum (children 1-8, males 9+, females 9+). */

data nhanes1 nhanes2 nhanes3;
  set nhanes;

  if (stratum = 1) then output nhanes1;
  if (stratum = 2) then output nhanes2;
  if (stratum = 3) then output nhanes3;
  run;

data init_parms1 (drop=stratum A1_age_14to18 A1_age_19to30 A1_age_31to50 A1_age_51to70 A1_age_71plus
                               A2_age_14to18 A2_age_19to30 A2_age_31to50 A2_age_51to70 A2_age_71plus)
     init_parms2 (drop=stratum A1_female A1_age_4to8 A2_female A2_age_4to8)
     init_parms3 (drop=stratum A1_female A1_age_4to8 A2_female A2_age_4to8);

  set init_parms;
  if (stratum = 1) then output init_parms1;
  if (stratum = 2) then output init_parms2;
  if (stratum = 3) then output init_parms3;
  run;

 /* delete unneeded data sets. */

proc datasets lib=work nolist;
  delete nhanes init_parms_f1 init_parms_f2 init_parms;
  run;
  quit;

 /* call macro NLMixed_Bivariate to fit nonlinear mixed model in each stratum. */

 /* in this example, we specify fixed values for parameters lambda1 (box-cox   */
 /* transformation for food1) and lambda2 (box-cox transformation for food 2). */
 /* Alternatively, one can let the macro estimate lambda1 and lambda2 by not   */
 /* specifying values for macro parameters "lambda1" and "lambda2".            */

title2 "Stratum 1 = Chilren 1-8";

%NLMixed_Bivariate (data           = nhanes1,
                    init_parms     = init_parms1,
                    subject        = id,
                    repeat         = repeat,
                    response1      = recall_food1,
                    response2      = recall_food2,
                    modeltype      = &modeltype,
                    covars_prob1   = repeat2 weekend female age_4to8,
                    covars_amt1    = repeat2 weekend female age_4to8,
                    covars_amt2    = repeat2 weekend female age_4to8,
                    lambda1        = &lambda11,
                    lambda2        = &lambda12,
                    replicate_var  = rndw,
                    nloptions      = &nloptions,
                    print          = Y,
                    ntitle         = 2);

data parms_b1;
  stratum = 1;
  set parms_b;
  run;

data pred_b1;
  stratum = 1;
  set pred_x_b;
  run;

title2 "Stratum 2 = Males 9+";

%NLMixed_Bivariate (data           = nhanes2,
                    init_parms     = init_parms2,
                    subject        = id,
                    repeat         = repeat,
                    response1      = recall_food1,
                    response2      = recall_food2,
                    modeltype      = &modeltype,
                    covars_prob1   = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    covars_amt1    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    covars_amt2    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    lambda1        = &lambda21,
                    lambda2        = &lambda22,
                    replicate_var  = rndw,
                    nloptions      = &nloptions,
                    print          = Y,
                    ntitle         = 2);

data parms_b2;
  stratum = 2;
  set parms_b;
  run;

data pred_b2;
  stratum = 2;
  set pred_x_b;
  run;

title2 "Stratum 3 = Females 9+";

%NLMixed_Bivariate (data           = nhanes3,
                    init_parms     = init_parms3,
                    subject        = id,
                    repeat         = repeat,
                    response1      = recall_food1,
                    response2      = recall_food2,
                    modeltype      = &modeltype,
                    covars_prob1   = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    covars_amt1    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    covars_amt2    = repeat2 weekend age_14to18 age_19to30 age_31to50 age_51to70 age_71plus,
                    lambda1        = &lambda31,
                    lambda2        = &lambda32,
                    replicate_var  = rndw,
                    nloptions      = &nloptions,
                    print          = Y,
                    ntitle         = 2);

data parms_b3;
  stratum = 3;
  set parms_b;
  run;

data pred_b3;
  stratum = 3;
  set pred_x_b;
  run;

 /* combine data sets of parameter estimates and predicted values. */

data parms_b;
  set parms_b1 parms_b2 parms_b3;
    by stratum;
  run;

data pred_b;
  set pred_b1 pred_b2 pred_b3;
    by stratum;
  run;

 /* keep minimum amount on consumption day. */

data parms_b;
  merge parms_b min_a;
    by stratum;
  run;

 /* save parameter estimates and predicted values. */

data outlib.parms_b_&foodvar1._&foodvar2;
  set parms_b;
  run;

data outlib.pred_b_&foodvar1._&foodvar2;
  set pred_b;
  run;
