/*************************************************************************/
/*************************************************************************/
/*                                                                       */
/*                 THE HEI-TODDLERS-2020 SCORING MACRO                   */                                               
/*                 (heitodl20.score.macro.sas)                           */
/*************************************************************************/
/*                    VERSION 1.0        12/04/2024                      */   
/*                                                                       */
/* This HEI-Toddlers-2020 (HEITODL20) macro is to be used to calculate 	 */
/*  densities and HEITODL20 component and total scores of complementary  */
/*  foods and beverages for subjects age 12 through 23 months.           */
/*								                                                       */	
/* The macro requires an input dataset with variables for each of        */
/*  the HEITODL20 components, noted below.                               */
/*								                                                       */	
/* Before applying the macro, please confirm that all input variables 	 */
/*  do not include energy or nutrient contributions from human milk, 	   */
/*  infant formula, nutritive/nutrition beverages, (e.g., Pediasure,   	 */
/*  Ensure), or other supplements. [The HEITODL20 variables reflects   	 */
/*  recommendations for complementary foods and beverages for toddlers 	 */
/*  age 12 through 23 months who are no longer receiving human milk or	 */
/*  infant formula. If toddlers are still receiving these foods, the 	   */
/*  HEITODL20 score should be calculated without energy or nutrient 	   */
/*  contributions from these foods.]                    		             */
/*								                                                       */	
/* The resulting dataset, which is named by the user, contains the       */
/*  same variables as the supplied dataset, and creates 27 new           */
/*  variables. These include:					                                   */
/*                                                                       */				    
/*   The densities (per 1000 kcal) or percent (of total calories)        */
/*    for each of the 13 HEITODL20 components.			                     */
/*                                                                       */
/*   Scores for the 13 components of the HEITODL20.                      */
/*                                                                       */
/*   The total HEITODL20 score, which is the sum of the                  */
/*    scores for the 13 components.                                      */
/*                                                                       */                                                                  
/* The syntax for calling the macro is:                                  */
/*                                                                       */
/* %HEITODL20   	 				                                               */
/* (indat=,kcal=,f_total=,fwholefrt=,vtotalleg=,vdrkgrleg=,g_whole=      */
/*    d_total=,pfallprotleg=,pfseaplantleg=,monopoly=,g_refined=,sodium=,*/
/*    add_sugars=,satfat=,outdat=);                                      */
/*                                                                       */
/*  where                                                                */
/*                                                                       */
/*   "indat"        * Specifies the dataset to be used.   		           */
/*                                                                       */
/*   ALL input variables below EXCLUDE human milk, infant formula, 	     */
/*   nutritive/nutrition beverages (e.g., Pediasure, Ensure), and other  */
/*   supplements. 							                                         */
/*									                                                     */
/*                                                                       */
/*   "kcal"           * Specifies calorie amount (excluding calories     */
/*                       from human milk, infant formula, and 		       */
/*		         nutritive/nutrition beverages).                           */
/*                                                                       */
/*   "f_total"        * Specifies the intake of total fruit in cup eq.   */
/*                                                                       */
/*   "fwholefrt"      * Specifies the intake of whole fruit in cup eq.   */
/*                                                                       */
/*   "vtotalleg"      * Specifies the intake of total veg plus           */
/*                       legumes in cup eq.                              */
/*                                                                       */
/*   "vdrkgrleg"      * Specifies the intake of dark green veg           */
/*                       plus legumes in cup eq.                         */
/*                                                                       */
/*   "g_whole"        * Specifies the intake of whole grain in oz. eq.   */
/*                                                                       */
/*   "d_total"        * Specifies the intake of total dairy in cup eq.   */
/*                                                                       */
/*   "pfallprotleg"   * Specifies the intake of total protein foods      */
/*                      (includes legumes) in oz. eq.                    */
/*                                                                       */
/*   "pfseaplantleg"  * Specifies the intake of seafood, fish and plant  */
/*                       protein foods (includes legumes) in oz. eq.     */
/*                                                                       */
/*   "monopoly"       * Specifies the grams of mono fat plus poly fat.   */
/*                                                                       */
/*   "g_refined"      * Specifies the intake of refined                  */
/*                       grain in oz. eq.                                */
/*                                                                       */
/*   "sodium"         * Specifies the mg of sodium.                      */
/*                                                                       */                                                                 
/*   "add_sugars"     * Specifies the intake of added sugars in tsp. eq. */
/*                                                                       */
/*   "satfat"         * Specifies the grams of saturated fat.            */
/*                                                                       */
/*   "outdat"         * Specifies the name of the resulting dataset.     */
/*								         */
/*                                                                       */                                                                 
/* Caution:  variable names "FARMIN", "FARMAX", "SODMIN",                */
/*   "SODMAX", "RGMIN", "RGMAX", "SFATMIN", "SFATMAX", "ADDSUGMIN",      */
/*   "ADDSUGMAX" are reserved for this macro.                            */
/*                                                                       */
/*                                                                       */
/*************************************************************************/
;

/*************************************************************************/
/* ALL dietary input variables EXCLUDE human milk, infant formula,     	 */
/*  nutritive/nutrition beverages (e.g., Pediasure, Ensure), and other 	 */
/*  supplements.                                                         */
/*************************************************************************/

%macro HEITODL20 (indat=,kcal=,f_total=,fwholefrt=,vtotalleg=,vdrkgrleg=,g_whole=,     
    d_total=,pfallprotleg=,pfseaplantleg=,monopoly=,g_refined=,sodium=,add_sugars=,satfat=,outdat=);

data &outdat (drop=FARMIN FARMAX SODMAX SODMIN RGMIN RGMAX SFATMIN SFATMAX ADDSUGMIN ADDSUGMAX);
  set &indat;

/*********************************************************************************************/
/*  This section of the macro is for scoring Adequacy components; higher 		                 */
/*  reported density-based intakes (variables containing "DEN") of these 		                 */
/*  components results in higher scores (up to the maximum points for that component).	     */
/*********************************************************************************************/

  IF &kcal > 0 then FRTDEN=&f_total/(&kcal/1000);
  HEITODL20_TOTALFRUIT=5*(FRTDEN/0.7);
  IF HEITODL20_TOTALFRUIT > 5 THEN HEITODL20_TOTALFRUIT=5;
  IF FRTDEN=0 THEN HEITODL20_TOTALFRUIT=0;	

  IF &kcal > 0 then WHFRDEN=&fwholefrt/(&kcal/1000);
  HEITODL20_WHOLEFRUIT=5*(WHFRDEN/0.3); 
  IF HEITODL20_WHOLEFRUIT > 5 THEN HEITODL20_WHOLEFRUIT=5;
  IF WHFRDEN=0 THEN HEITODL20_WHOLEFRUIT=0;
  
  IF &kcal > 0 then VEGDEN=&vtotalleg/(&kcal/1000);
  HEITODL20_TOTALVEG=5*(VEGDEN/0.9);
  IF HEITODL20_TOTALVEG > 5 THEN HEITODL20_TOTALVEG=5;
  IF VEGDEN=0 THEN HEITODL20_TOTALVEG=0;

  IF &kcal > 0 then GRBNDEN=&vdrkgrleg/(&kcal/1000);
  HEITODL20_GREEN_AND_BEAN=5*(GRBNDEN/0.1);
  IF HEITODL20_GREEN_AND_BEAN > 5 THEN HEITODL20_GREEN_AND_BEAN=5;
  IF GRBNDEN=0 THEN HEITODL20_GREEN_AND_BEAN=0;

  IF &kcal > 0 then WGRNDEN=&g_whole/(&kcal/1000);
  HEITODL20_WHOLEGRAIN=10*(WGRNDEN/1.5);
  IF HEITODL20_WHOLEGRAIN > 10 THEN HEITODL20_WHOLEGRAIN=10;
  IF WGRNDEN=0 THEN HEITODL20_WHOLEGRAIN=0;

  IF &kcal > 0 then DAIRYDEN=&d_total/(&kcal/1000);
  HEITODL20_TOTALDAIRY=10*(DAIRYDEN/2.0);
  IF HEITODL20_TOTALDAIRY > 10 THEN HEITODL20_TOTALDAIRY=10;
  IF DAIRYDEN=0 THEN HEITODL20_TOTALDAIRY=0;

  IF &kcal > 0 then PROTDEN=&pfallprotleg/(&kcal/1000);
  HEITODL20_TOTPROT=5*(PROTDEN/2.0);
  IF HEITODL20_TOTPROT > 5 THEN HEITODL20_TOTPROT=5;
  IF PROTDEN=0 THEN HEITODL20_TOTPROT=0;

  IF &kcal > 0 then SEAPLDEN=&pfseaplantleg/(&kcal/1000);
  HEITODL20_SEAPLANT_PROT=5*(SEAPLDEN/0.5);
  IF HEITODL20_SEAPLANT_PROT > 5 THEN HEITODL20_SEAPLANT_PROT=5;
  IF SEAPLDEN=0 THEN HEITODL20_SEAPLANT_PROT=0;


/**************************************************************************************************/
/*  The Fatty Acid component is calculated as a ratio; variables listed as "MAX" and "MIN"        */
/*  here relate to the ratio of unsaturated to saturated fat; thus the "MIN" value corresponds 	  */
/*  to the minimum point standard and the "MAX" value corresponds to the maximum point standard.  */
/**************************************************************************************************/


  IF &satfat > 0 THEN FARATIO=&monopoly/&satfat;
  FARMIN=0.9;
  FARMAX=1.5;
  if &satfat=0 and &monopoly=0 then HEITODL20_FATTYACID=0;
    else if &satfat=0 and &monopoly > 0 then HEITODL20_FATTYACID=10;
    else if FARATIO >= FARMAX THEN HEITODL20_FATTYACID=10;
    else if FARATIO <= FARMIN THEN HEITODL20_FATTYACID=0;
    else HEITODL20_FATTYACID=10* ( (FARATIO-FARMIN) / (FARMAX-FARMIN) );


/************************************************************************************************/
/*  The following section of the macro is for scoring Moderation components; higher reported    */
/*  intake of these food components results in LOWER scores while lower reported intake results */
/*  in HIGHER scores (up to the maximum points for that component). Variables below labeled     */
/*  "MIN" or "MAX" relate to intake amounts; thus "MIN" values correspond to the maximum point  */
/*  standard and "MAX" values correspond to the minimum point standard for that component.      */
/************************************************************************************************/


  IF &kcal > 0 then RGDEN=&g_refined/(&kcal/1000);
  RGMIN=1.5;
  RGMAX=3.4;
  IF RGDEN <= RGMIN THEN HEITODL20_REFINEDGRAIN=10;
    ELSE IF RGDEN >= RGMAX THEN HEITODL20_REFINEDGRAIN=0;
    ELSE HEITODL20_REFINEDGRAIN=10 - ( 10* (RGDEN-RGMIN) / (RGMAX-RGMIN) ); 

  IF &kcal > 0 then SODDEN=&sodium/&kcal;
  SODMIN=1.1;
  SODMAX=1.7;
  IF SODDEN <= SODMIN THEN HEITODL20_SODIUM=10;
    ELSE IF SODDEN >= SODMAX THEN HEITODL20_SODIUM=0;
    ELSE HEITODL20_SODIUM=10 - (10 * (SODDEN-SODMIN) / (SODMAX-SODMIN) );

  IF &kcal > 0 then ADDSUG_PERC=100*(&add_sugars*16/&kcal); 
  ADDSUGMIN=0;
  ADDSUGMAX=13.8; 
  IF ADDSUG_PERC >= ADDSUGMAX THEN HEITODL20_ADDSUG=0;
    ELSE IF ADDSUG_PERC <= ADDSUGMIN THEN HEITODL20_ADDSUG=10;
    ELSE HEITODL20_ADDSUG= 10 - ( 10* (ADDSUG_PERC-ADDSUGMIN) / (ADDSUGMAX-ADDSUGMIN) );

  IF &kcal > 0 then SFAT_PERC=100*(&satfat*9/&kcal); 
  SFATMIN=12.2;
  SFATMAX=18.2;
  IF SFAT_PERC >= SFATMAX THEN HEITODL20_SFAT=0;
    ELSE IF SFAT_PERC <= SFATMIN THEN HEITODL20_SFAT=10;
    ELSE HEITODL20_SFAT= 10 - ( 10* (SFAT_PERC-SFATMIN) / (SFATMAX-SFATMIN) );


IF &kcal=0 THEN DO;
  HEITODL20_TOTALFRUIT=0; HEITODL20_WHOLEFRUIT=0; HEITODL20_TOTALVEG=0; HEITODL20_GREEN_AND_BEAN=0;  HEITODL20_WHOLEGRAIN=0; HEITODL20_TOTALDAIRY=0;
  HEITODL20_TOTPROT=0;  HEITODL20_SEAPLANT_PROT=0; HEITODL20_FATTYACID=0; HEITODL20_REFINEDGRAIN=0; HEITODL20_SODIUM=0;  HEITODL20_ADDSUG=0; HEITODL20_SFAT=0;
  END;


HEITODL20_TOTAL_SCORE = HEITODL20_TOTALFRUIT + HEITODL20_WHOLEFRUIT + HEITODL20_TOTALVEG + HEITODL20_GREEN_AND_BEAN + HEITODL20_WHOLEGRAIN + HEITODL20_TOTALDAIRY +
  HEITODL20_TOTPROT + HEITODL20_SEAPLANT_PROT + HEITODL20_FATTYACID + HEITODL20_REFINEDGRAIN + HEITODL20_SODIUM  + HEITODL20_ADDSUG + HEITODL20_SFAT;


LABEL HEITODL20_TOTAL_SCORE='TOTAL HEITODL20 SCORE'
      HEITODL20_TOTALFRUIT='HEITODL20 COMPONENT TOTAL FRUIT'
      HEITODL20_WHOLEFRUIT='HEITODL20 COMPONENT WHOLE FRUIT'
      HEITODL20_TOTALVEG='HEITODL20 COMPONENT TOTAL VEGETABLES'
      HEITODL20_GREEN_AND_BEAN='HEITODL20 COMPONENT GREENS AND BEANS'
      HEITODL20_WHOLEGRAIN='HEITODL20 COMPONENT WHOLE GRAINS'
      HEITODL20_TOTALDAIRY='HEITODL20 COMPONENT DAIRY'
      HEITODL20_TOTPROT='HEITODL20 COMPONENT TOTAL PROTEIN FOODS'
      HEITODL20_SEAPLANT_PROT='HEITODL20 COMPONENT SEAFOOD AND PLANT PROTEIN'
      HEITODL20_FATTYACID='HEITODL20 COMPONENT FATTY ACID RATIO'
      HEITODL20_REFINEDGRAIN='HEITODL20 COMPONENT REFINED GRAINS'
      HEITODL20_SODIUM='HEITODL20 COMPONENT SODIUM'
      HEITODL20_ADDSUG='HEITODL20 COMPONENT ADDED SUGAR'
      HEITODL20_SFAT='HEITODL20 COMPONENT SAT FAT'

      FRTDEN='DENSITY OF TOTAL FRUIT PER 1000 KCAL'
      WHFRDEN='DENSITY OF WHOLE FRUIT PER 1000 KCAL'
      VEGDEN='DENSITY OF TOTAL VEGETABLES PER 1000 KCAL'
      GRBNDEN='DENSITY OF DARK GREEN VEG AND BEANS PER 1000 KCAL'
      WGRNDEN='DENSITY OF WHOLE GRAIN PER 1000 KCAL'
      DAIRYDEN='DENSITY OF DAIRY PER 1000 KCAL'
      PROTDEN='DENSITY OF TOTAL PROTEIN PER 1000 KCAL'
      SEAPLDEN='DENSITY OF SEAFOOD AND PLANT PROTEIN PER 1000 KCAL'
      FARATIO='FATTY ACID RATIO'
      RGDEN='DENSITY OF REFINED GRAINS PER 1000 KCAL'
      SODDEN='DENSITY OF SODIUM PER 1000 KCAL'
      ADDSUG_PERC='PERCENT OF CALORIES FROM ADDED SUGAR'
      SFAT_PERC='PERCENT OF CALORIES FROM SAT FAT'
      ;

run;

%mend HEITODL20;


/*******************************************************************/
/*  END OF THE HEITODL20 MACRO                                     */
/*******************************************************************/
