check_type <- function(x, data, name="type") {

  nms   <- c("Demographics", "Dietary", "Examination", "Laboratory", "Questionnaire") 
  valid <- toupper(substr(nms, 1, 2))
  names(valid) <- nms
  if (data == name_nnyfs()) {
    tmp   <- !(valid %in% "LA")
    valid <- valid[tmp]
  }
  
  if (!length(x)) return(valid)
  
  str <- paste0(nms, collapse=", ")
  if (!is.vector(x)) stop(paste0("ERROR: ", name, " must be a vector with values ", str))
  if (!is.character(x)) stop(paste0("ERROR: ", name, " must be a vector with values ", str))
  
  x   <- unique(removeWhiteSpace(x))
  len <- nchar(x)
  if (any(len < 2)) stop(paste0("ERROR: ", name, " must be a vector with values ", str))
  x <- toupper(substr(x, 1, 2))
  tmp <- !(x %in% valid)
  if (any(tmp)) stop(paste0("ERROR: ", name, " must be a vector with values ", str))
  
  tmp <- valid %in% x
  ret <- valid[tmp]
  
  ret
  
}

check_subject.seqn <- function(x, valid, name="subject.seqn") {

  if (!length(x)) return(NULL)
  if (!is.vector(x)) stop(paste0("ERROR: ", name, " must be a vector of subject SEQN ids"))
  x <- removeWhiteSpace(unique(x))
  
  miss <- NULL
  tmp  <- !(x %in% valid)
  if (any(tmp)) {
    miss <- x[tmp]
    x    <- x[!tmp] 
    #err <- paste0(x[tmp], collapse=", ")
    #print(err)
    #warning(paste0("The above subjects do not have accelerometer data."))
  }

  list(seqn=x, missing=miss)

}

check_data <- function(x, name="data") {

  if (!isString(x)) stop(paste0("ERROR: ", name, " must be a character string"))
  x  <- unique(toupper(x))
  nx <- length(x)
  valid <- names(getRawAccDataURLs())
  vstr  <- paste0(valid, collapse=", ")
  if (nx != 1) stop(paste0("ERROR: ", name, " must be one of ", vstr ))
  tmp <- !(x %in% valid)
  if (any(tmp)) stop(paste0("ERROR: ", name, " must be one of ", vstr ))
  x

}

check_zipcmd <- function(x, name="zipcmd") {

  ret <- NULL
  if (OSisWindows()) {
    if (!isString(x)) {
      stop(paste0("ERROR: zipcmd must be the path to the 7-zip or WinZip software"))
    }
    ret <- checkDirSep(x)
  }
  ret
}

checkDirSep <- function(x) {

  x <- removeWhiteSpace(x)
  x <- gsub("\\", "/", x, fixed=TRUE)
  x

}

check_destDir <- function(x, name="destDir") {

  if (!isString(x)) stop(paste0("ERROR: ", name, " must be a character string"))
  x <- removeWhiteSpace(x)
  if (!dir.exists(x)) stop(paste0("ERROR: ", x, " does not exist"))
  if (file.access(x, mode=2)) stop(paste0("ERROR: ", x, " does not have write permission"))

  x <- gsub("\\", "/", x, fixed=TRUE)
  x <- checkForSep(x)
  x

}

check_files <- function(files, data, name="files", len=0) {

  nfiles <- length(files)
  if (nfiles) {
    if (!is.character(files) || !is.vector(files)) {
      stop(paste0("ERROR: ", name, " must be a character vector"))
    }
  }

  if (len && (nfiles != 1)) stop(paste0("ERROR: ", name, " must be a single file"))
  files  <- unique(toupper(removeWhiteSpace(files)))
  x      <- getDataFileInfo(data)

  dnames <- toupper(x[, infoNameCol(), drop=TRUE]) 
  dfiles <- toupper(x[, infoFileCol(), drop=TRUE])
  ind1   <- match(files, dnames)
  ind2   <- match(files, dfiles) 
  tmp    <- is.na(ind1)
  if (any(tmp)) ind1[tmp] <- ind2[tmp]
  tmp    <- is.na(ind1)
  if (any(tmp)) {
    err <- files[tmp]
    print(err)
    stop("ERROR: the above files are not valid")   
  } 
  ret <- x[ind1, , drop=FALSE] 
  ret 
  
}

check_save.type <- function(x, name="save.type") {

  valid <- c("xpt", "rda", "csv")
  vstr  <- paste0(valid, collapse=", ")
  if (!isString(x)) stop(paste0("ERROR: ", name, " must be one of ", vstr))
  x <- tolower(removeWhiteSpace(x))
  if (!(x %in% valid)) stop(paste0("ERROR: ", name, " must be one of ", vstr))
  x

}

check_data_object <- function(x, name, vars="SEQN") {

  if (is.data.frame(x)) {
    check_data_frame(x, name, vars=vars)
    ret <- x
  } else if (isString(x)) {
    if (!file.exists(x)) stop(paste0("ERROR: ", x, " does not exist"))
    ret <- read_file(x)
    check_data_frame(ret, name, vars=vars)
  } else {
    stop(paste0("ERROR: ", name, " must be a data frame or file"))
  }
  ret 
}

check_data_frame <- function(x, name, vars=NULL) {

  if (!is.data.frame(x)) stop(paste0("ERROR: ", name, " must be a data frame"))
  if (!length(x)) stop(paste0("ERROR: ", name, " contains no columns"))
  if (!nrow(x)) stop(paste0("ERROR: ", name, " contains no rows"))
  if (length(vars)) {
    tmp <- !(vars %in% colnames(x))
    if (any(tmp)) {
      miss <- vars[tmp]
      n    <- length(miss)
      str  <- ""
      str2 <- paste0(miss, collapse=", ")
      if (n > 1) str <- "s"
      msg <- paste0("ERROR: ", name, " is missing column", str, " ", str2)
      stop(msg) 
    }
  }
  NULL
}
