getBaseFilesFromBz2 <- function(f) {

  ret <- NULL
  if (OSisUnix()) {
    ret <- getBaseFilesFromBz2_unix(f)
  } else if (OSisWindows()) {
    # Figure out how to get file names. 7z l returns the tar file name
    ret <- NULL
  } else {
    ret <- NULL
  }
  ret
}

getBaseFilesFromBz2_unix <- function(f) {

  x <- system(paste0('tar -tf "', f, '"'), intern=TRUE)
  if (length(x)) {
    x <- basename(x)
  } else {
    x <- NULL
  }
  x

}

extractFilesFromBz2 <- function(bz2File, destdir, zipcmd, DEBUG=0) {

  if (OSisUnix()) {
    extractFilesFromBz2_unix(bz2File, destdir) 
  } else if (OSisWindows()) {
    extractFilesFromBz2_windows(bz2File, destdir, zipcmd, DEBUG=DEBUG) 
  } else {
    # Try unix method
    extractFilesFromBz2_unix(bz2File, destdir) 
  }
  NULL
}

extractFilesFromBz2_unix <- function(bz2File, destdir) {

  cmd <- paste0('tar -xf "', bz2File, '" --directory="', destdir, '"')
  system(cmd)
  NULL
}

is7zip <- function(zipcmd) {

  prog <- tolower(basename(zipcmd))
  ret  <- grepl("7z", prog, fixed=TRUE)
  ret 
}

extractFilesFromBz2_windows <- function(bz2File, destdir, zipcmd, DEBUG=0) {

  if (is7zip(zipcmd)) {
    ret <- extractFilesFromBz2_windows_7zip(bz2File, destdir, zipcmd, DEBUG=DEBUG)
  } else {
    ret <- extractFilesFromBz2_windows_winzip(bz2File, destdir, zipcmd, DEBUG=DEBUG)
  }
  ret
}

extractFilesFromBz2_windows_7zip <- function(bz2File, destdir, zipcmd, DEBUG=0) {

  rc   <- NULL
  prog <- basename(zipcmd)

  f2  <- gsub(".bz2", "", basename(bz2File), fixed=TRUE)
  f2  <- paste0(destdir, f2)
  cmd <- paste0(prog, ' e ', '"', bz2File, '"', ' -o', '"', destdir, '"', ' -y')
  if (DEBUG) {
    print(destdir)
    print(f2)
    print(cmd)
  }
  callOS(cmd)
  if (!file.exists(f2)) {
    print(cmd)
    stop("ERROR 1 in extractFilesFromBz2_windows_7zip")
  }
  cmd <- paste0(prog, ' x ', '"', f2,      '"', ' -o', '"', destdir, '"', ' -y')
  callOS(cmd)
  if (file.exists(f2)) file.remove(f2)
  
  NULL
}

extractFilesFromBz2_windows_winzip <- function(bz2File, destdir, zipcmd, DEBUG=0) {

  rc   <- NULL
  #prog <- basename(zipcmd)
  prog <- "WZUNZIP"

  f2  <- gsub(".bz2", "", basename(bz2File), fixed=TRUE)
  f2  <- paste0(destdir, f2)
  cmd <- paste0(prog, ' -e -o -jhrs ', '"', bz2File, '"', ' ', '"', destdir, '"')
  if (DEBUG) {
    print(destdir)
    print(f2)
    print(cmd)
  }
  callOS(cmd)
  if (!file.exists(f2)) {
    print(cmd)
    stop("ERROR 1 in extractFilesFromBz2_windows_winzip")
  }
  cmd <- paste0(prog, ' -e -o -jhrs ', '"', f2,      '"', ' ', '"', destdir, '"')

  callOS(cmd)
  if (file.exists(f2)) file.remove(f2)
  
  NULL
}

setPathEnvVar_windows <- function(zipcmd, DEBUG=0) {

  rc   <- NULL
  dir  <- dirname(zipcmd)
  prog <- basename(zipcmd)
  if (DEBUG) {
    print(paste0("dir=", dir))
    print(paste0("prog=", prog))
  }
  if (dir == ".") return(rc)

  p <- Sys.getenv("PATH")
  p2 <- paste(p, ";", dir, ";", sep="")
  rc <- Sys.setenv(PATH=p2)
  if (DEBUG) {
    print(paste0("Sys.getenv=", p))
    print(paste0("p2=", p))
    print(paste0("Sys.setenv return code=", rc))
  }
  rc
}

compressFiles <- function(files=NULL, dir=NULL, with.substr=NULL, zipcmd=NULL, DEBUG=0) {

  if (!length(files)) {
    if (!length(dir)) stop("ERROR 1 in compressFiles")
    files <- list.files(dir, full.names=TRUE)
  }
  tmp  <- file.exists(files)
  miss <- !tmp
  if (any(miss)) {
    print(files[miss])
    message("NOTE: the files listed above were not found")
  }
  files  <- files[tmp]
  if (!length(files)) return(NULL)

  if (!is.null(with.substr)) {
    tmp   <- grepl(with.substr, files, fixed=TRUE)
    files <- files[tmp] 
  }
  if (!length(files)) return(NULL)

  # Do not try to re-compress a .gz file
  len      <- nchar(files)
  tmp      <- tolower(substr(files, len-2, len)) == ".gz"
  files    <- files[!tmp]
  if (!length(files)) return(NULL)

  message("Compressing files")
  if (OSisUnix()) {
    compressFiles_unix(files) 
  } else if (OSisWindows()) {
    compressFiles_windows(files, zipcmd, DEBUG=DEBUG)  
  } else {
    # Try unix method
    compressFiles_unix(files) 
  }
  NULL
}

compressFiles_unix <- function(files) {

  nfiles <- length(files)
  if (nfiles) {
    for (i in 1:nfiles) system(paste0('gzip "', files[i], '"')) 
  }  

  NULL
}

compressFiles_windows <- function(files, zipcmd, DEBUG=0) {

  nfiles <- length(files)
  if (!nfiles) return(NULL)

  if (is7zip(zipcmd)) {
    ret <- compressFiles_windows_7zip(files, zipcmd, DEBUG=DEBUG)
  } else {
    ret <- compressFiles_windows_winzip(files, zipcmd, DEBUG=DEBUG)
  }
  ret
}

compressFiles_windows_7zip <- function(files, zipcmd, DEBUG=0) {

  rc   <- NULL
  prog <- basename(zipcmd)
  for (i in 1:length(files)) {
    out <- paste0(files[i], ".gz")
    cmd <- paste0(prog, ' a ', '"', out, '"', ' ', '"', files[i], '"', ' -sdel')
    if (DEBUG) print(cmd)
    ret <- callOS(cmd, intern=TRUE)
    if (DEBUG) print(ret)
  } 
 
  NULL
}

compressFiles_windows_winzip <- function(files, zipcmd, DEBUG=0) {

  rc   <- NULL
  #prog <- basename(zipcmd)
  prog <- "WZZIP"

  for (i in 1:length(files)) {
    out <- paste0(files[i], ".zip")
    cmd <- paste0(prog, ' -a+ -jhrs ', '"', out, '"', ' ', '"', files[i], '"')
    if (DEBUG) print(cmd)
    ret <- callOS(cmd, intern=TRUE)
    if (DEBUG) print(ret)
    if (file.exists(files[i])) file.remove(files[i])
  } 
 
  NULL
}

