removeWhiteSpace <- function(str, leading=1, trailing=1) {

  if ((leading) && (trailing)) {
    ret <- gsub("^\\s+|\\s+$", "", str, perl=TRUE)
  } else if (leading) {
    ret <- gsub("^\\s+", "", str, perl=TRUE)
  } else if (trailing) {
    ret <- gsub("\\s+$", "", str, perl=TRUE)
  } else {
    ret <- str
  }

  ret

}

removeFiles <- function(ff) {

  if (!length(ff)) return(NULL)
  tmp <- file.exists(ff)
  if (any(tmp)) {
    dir <- unique(dirname(ff))
    if (length(dir) == 1) message(paste0("Deleting files in ", dir)) 
    file.remove(ff[tmp])
  }
  NULL
}

getVecFromStr <- function(string, delimiter="|") {

  # string       String to break apart. No default
  # delimiter    Delimiter used in string. The default is "|". 

  strsplit(string, delimiter, fixed=TRUE)[[1]]

}

isString <- function(x) {
  (length(x) == 1) && is.character(x)
}

checkForSep <- function(x) {

  len <- nchar(x)
  if (!len) return("./")
  c1 <- substr(x, len-1, len)
  if (c1 != "/") x <- paste0(x, "/")
  x

}

parseDelimVec0 <- function(vec, sep, ncol, numeric=0) {

  mat <- unlist(strsplit(vec, sep, fixed=TRUE))
  if (length(mat) != length(vec)*ncol) {
    print("ERROR: check ncol or if some elements of the vector are missing delimiters")
    stop("ERROR: with vec")
  }
  if (numeric) mat <- as.numeric(mat)
  mat <- matrix(mat, byrow=TRUE, ncol=ncol)
  return(mat)

  mat   

} # END: parseDelimVec0

default.list <- function(inList, names, default, error=NULL,
                         checkList=NULL) {

  # inList      List
  # names       Vector of names of items in inList
  # default     List of default values to assign if a name is not found
  #             The order of default must be the same as in names.
  # error       Vector of TRUE/FALSE if it is an error not to have the
  #             name in the list. 
  #             The default is NULL
  # checkList   List of valid values for each name.
  #             Use NA to skip a list element.
  #             The default is NULL

  n1 <- length(names)
  n2 <- length(default)
  if (n1 != n2) stop("ERROR: in calling default.list")

  if (is.null(error)) {
    error <- rep(0, times=n1)
  } else if (n1 != length(error)) {
    stop("ERROR: in calling default.list")
  }

  if (!is.null(checkList)) {
    if (n1 != length(checkList)) stop("ERROR: in calling default.list")
    checkFlag <- 1
  } else {
    checkFlag <- 0
  } 

  if (is.null(inList)) inList <- list()

  listNames <- names(inList)
  for (i in 1:n1) {
    if (!(names[i] %in% listNames)) {
      if (!error[i]) {
        inList[[names[i]]] <- default[[i]]
      } else {
        temp <- paste("ERROR: the name ", names[i], " was not found", sep="")
        stop(temp)
      }
    } else if (checkFlag) {
      temp <- checkList[[i]]
      if (!all(is.na(temp))) {
        if (!all(inList[[names[i]]] %in% checkList[[i]])) {
          temp <- paste("ERROR: the name '", names[i], 
                      "' has an invalid value", sep="")
          stop(temp)
        }
      }
    }
  }

  inList

} # END: default.list



getOS <- function() {
  tolower(.Platform$OS.type)
}

OSisWindows <- function() {
  getOS() == "windows"
}

OSisUnix <- function() {
  getOS() == "unix"
}

callOS <- function(command, intern=FALSE) {

  if (OSisWindows()) {
    ret <- shell(command, intern=intern)
  } else {
    ret <- system(command, intern=intern)
  }
  ret

} 

read_xport <- function(f) {

  x <- read.xport(f)
  x

}

read_file <- function(f, type=NULL) {

  if (!length(type)) type <- getFileExt(f)
  if (is.null(type)) stop(paste0("ERROR: no file extension in file ", f))

  if (type == "csv.zip") {
    # Special case, file was compressed with WinZip
    f2  <- substr(f, 1, nchar(f)-4) # Remove .zip
    fid <- unz(f, f2)
    x   <- read.table(fid, header=1, sep=",", stringsAsFactors=FALSE)
    close(fid)
  } else if (type == "xpt") {
    ret <- read_xport(f)
  } else if (type == "csv") {
    ret <- read.table(f, header=1, sep=",", stringsAsFactors=FALSE)
  } else if (type == "rda") {
    tmp <- load(f)
    ret <- eval(parse(text=tmp))
  } else {
    stop(paste0("ERROR: cannot read file ", f))
  }
  ret

}

getFileExt <- function(f) {

  zipFlag <- FALSE

  f <- tolower(removeWhiteSpace(basename(f)))
  # Check for .gz
  m <- nchar(f)
  if (m > 2) {
    if (substr(f, m-2, m) == ".gz") f <- gsub(".gz", "", f, fixed=TRUE)
  }
  # Check for .zip (special case)
  if (m > 3) {
    if (substr(f, m-3, m) == ".zip") {
      zipFlag <- TRUE 
      f       <- gsub(".zip", "", f, fixed=TRUE)
    }
  }

  x <- getVecFromStr(f, delimiter=".")
  n <- length(x)
  if (!n) return(NULL)
  ret <- tolower(x[n])
  if (zipFlag) ret <- paste0(ret, ".zip")
  ret

}

saveXptFileAs <- function(f, out, ext="csv") {

  data <- read_xport(f)
  if (ext == "csv") {
    write.table(data, file=out, sep=",", quote=FALSE, row.names=FALSE, col.names=TRUE)
  } else if (ext == "rda") {
    save(data, file=out)
  } else {
    stop("ERROR 1")
  }
  NULL

}

merge2 <- function(x1, x2, id1="SEQN", id2="SEQN") {

  ids1  <- removeWhiteSpace(x1[, id1, drop=TRUE])
  ids2  <- removeWhiteSpace(x2[, id2, drop=TRUE])
  uids1 <- unique(ids1)
  uids2 <- unique(ids2)
  tmp12 <- uids1 %in% uids2
  tmp21 <- uids2 %in% uids1

  if (!any(tmp12)) {
    # Distinct subject ids, merge rows
    ret <- mergeRowsDistinctSubs(x1, x2)
  } else {
    # Overlapping subjects
    if (all(tmp12) && all(tmp21)) {
      # Same subjects, merge columns
      ret <- mergeColsSameSubs(x1, x2, ids1, ids2)
    } else {
      # Not all the same subjects
      x1 <- addInMissingRowsToX1(x1, x2, uids1, uids2, idv1=id1)
      x2 <- addInMissingRowsToX1(x2, x1, uids2, uids1, idv1=id2)

      # Now x1 and x2 have the same subs
      ids1 <- removeWhiteSpace(x1[, id1, drop=TRUE])
      ids2 <- removeWhiteSpace(x2[, id2, drop=TRUE])
      ret  <- mergeColsSameSubs(x1, x2, ids1, ids2)
    }
  }
  rownames(ret) <- NULL
  ret
}

mergeColsSameSubs <- function(x1, x2, ids1, ids2) {

  tmp1 <- duplicated(ids1)
  tmp2 <- duplicated(ids2)
  sum1 <- sum(tmp1)
  sum2 <- sum(tmp2)
  if (sum1 && sum2) stop("ERROR: current version cannot merge this data due to duplicated ids in both sets of data")  
  if (sum1) {
    ret  <- x1
    x    <- x2
    rows <- match(ids1, ids2) 
  } else {
    ret  <- x2
    x    <- x1
    rows <- match(ids2, ids1) 
  }
  if (any(is.na(rows))) stop("INTERNAL CODING ERROR in mergeColsSameSubs")
  cx  <- colnames(x)
  tmp <- !(cx %in% colnames(ret))
  add <- cx[tmp]
  if (!length(add)) {
    warning("Same column names, data will not be merged")
    return(ret)
  }
  ret <- cbind(ret, x[rows, add, drop=FALSE])
  ret

}

mergeRowsDistinctSubs <- function(x1, x2) {

  cx1      <- colnames(x1)
  cx2      <- colnames(x2)
  sameCols <- FALSE
  if ((length(cx1) == length(cx2)) && all(cx1 %in% cx2)) sameCols <- TRUE
  if (!sameCols) {
    x1 <- addInMissingColsToX1(x1, x2)
    x2 <- addInMissingColsToX1(x2, x1)
  }
  ret <- rbind(x1, x2[, colnames(x1), drop=FALSE])

  ret
}

addInMissingColsToX1 <- function(x1, x2) {

  cx1 <- colnames(x1)
  cx2 <- colnames(x2)
  tmp <- !(cx2 %in% cx1)
  if (!any(tmp)) return(x1)
  miss       <- cx2[tmp]
  x1[, miss] <- NA
  x1

}

addInMissingRowsToX1 <- function(x1, x2, uids1, uids2, idv1="SEQN") {

  tmp  <- !(uids2 %in% uids1)
  if (!any(tmp)) return(x1)
  miss          <- uids2[tmp]
  cx            <- colnames(x1)
  add           <- data.frame(matrix(data=NA, nrow=length(miss), ncol=length(cx)))
  colnames(add) <- cx      
  if (!(idv1 %in% cx)) stop("INTERNAL CODING ERROR in addInMissingRowsToX1")
  add[, idv1]   <- miss
  x1            <- rbind(x1, add)
  x1

}
